package ru.yandex.chemodan.app.cvdemo2.admin;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.TimeZone;

import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchClient;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.auth.YateamAuthUtils;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUser;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistry;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

/**
 * @author tolmalev
 */
@ActionContainer
public class CvSelectorAdminPage {
    private final YandexStaffUserRegistry staffRegistry;
    private final DiskSearchClient diskSearchClient;
    private final MpfsClient mpfsClient;
    private final Cvi2tProcessor cvi2tProcessor;

    public CvSelectorAdminPage(YandexStaffUserRegistry staffRegistry, DiskSearchClient diskSearchClient, MpfsClient mpfsClient, Cvi2tProcessor cvi2tProcessor) {
        this.staffRegistry = staffRegistry;
        this.diskSearchClient = diskSearchClient;
        this.mpfsClient = mpfsClient;
        this.cvi2tProcessor = cvi2tProcessor;
    }

    @ZAction(defaultAction = true, file = "cv_selector.xsl")
    @Path("/cv-selector")
    public SelectorResultPojo makeSelection(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam(value = "uid", required = false) String uid,
            @RequestParam(value = "limit", required = false) Option<Integer> limitO,
            @RequestParam(value = "from", required = false) String from,
            @RequestParam(value = "to", required = false) String to,

            @RequestParam(value = "newBeautyLimit", required = false) Option<Double> newBeautyLimitO,
            @RequestParam(value = "maxBeauty", required = false) Option<Double> maxBeautyO,
            @RequestParam(value = "matchLimit", required = false) Option<Long> matchLimitO,

            @RequestParam(value = "text", required = false) Option<String> textO,

            @RequestParam(value = "sortByTime", required = false) Option<Boolean> sortByEtimeO
    ) {
        uid = StringUtils.defaultIfEmpty(uid, "").trim();
        from = StringUtils.defaultIfEmpty(from, "2018-06-01").trim();
        to = StringUtils.defaultIfEmpty(to, "2018-09-01").trim();

        int limit = limitO.getOrElse(100);
        double newBeautyLimit = newBeautyLimitO.getOrElse(1.0);
        double maxBeauty = maxBeautyO.getOrElse(10000.0);
        long matchLimit = matchLimitO.getOrElse(0L);

        boolean sortByEtime = sortByEtimeO.getOrElse(false);

        String text = textO.getOrElse("Море");

        ListF<String> words = Cf.x(text.split(",")).map(StringUtils::trim).filter(StringUtils::isNotBlank);

        Tuple2List<String, Option<Integer>> wordsWithLimits = words.toTuple2List(word -> {
            if (word.contains("(") && word.contains(")")) {
                String readWord = StringUtils.substringBefore(word, "(");
                String limitStr = StringUtils.substringBefore(StringUtils.substringAfter(word, "("), ")");

                return Tuple2.tuple(readWord, Cf.Integer.parseSafe(limitStr));
            } else {
                return Tuple2.tuple(word, Option.empty());
            }
        });

        String firstWord = wordsWithLimits.get1().first();

        if (uid.isEmpty()) {
            Option<String> loginO = YateamAuthUtils.getLoginFromAttributeO(reqX);
            if (loginO.isPresent()) {
                String internalLogin = loginO.getOrThrow(() -> new RuntimeException("Not authorized"));

                YandexStaffUser user = staffRegistry.findByInternalLogin(internalLogin)
                        .getOrThrow(() -> new RuntimeException(internalLogin + ": логин на staff не привязан"));

                uid = Long.toString(user.externalUid.getUid());
            } else {
                uid = "50273844";
            }
        }

        ListF<DiskSearchFileInfo> searchFilesInfo = diskSearchClient.findPhotosWithBeauty(
                PassportUid.cons(Long.parseLong(uid)),
                new InstantInterval(parseDate(from), parseDate(to)),
                Option.of("cool-lenta"),
                0, 100000
        ).hitsArray;

        Tuple2List<DiskSearchFileInfo, Long> selected = searchFilesInfo
                .filter(file -> file.getBeautiful2().isPresent() && file.getBeautiful2().get() > newBeautyLimit)
                .filter(file -> file.getBeautiful2().get() < maxBeauty)
                .filter(file -> {
                    if (!wordsWithLimits.get2().exists(Option::isPresent)) {
                        return true;
                    }

                    boolean foundGood = false;
                    for (Tuple2<String, Option<Integer>> tuple2 : wordsWithLimits) {
                        if (tuple2._2.isPresent() && cvi2tProcessor.dotProduct(file.geti2tVector(), tuple2._1) >= tuple2._2.get()) {
                            foundGood = true;
                        }
                    }
                    return foundGood;
                })
                .zipWith(file -> cvi2tProcessor.dotProduct(file.geti2tVector(), firstWord))
                .sortedBy2Desc()
                .filterBy2(match -> match >= matchLimit)
                .take(limit);

        String finalUid = uid;
        MapF<String, MpfsFileInfo> mpfsInfoByResourceId = selected.get1().paginate(50).flatMap(page -> {
            ListF<String> resourceIds = page.map(item -> finalUid + ":" + item.getId());
            return mpfsClient.bulkInfoByResourceIds(MpfsUser.of(finalUid), resourceIds, Cf.list("/disk", "/photounlim"));
        }).toMapMappingToKey(info -> finalUid + ":" + info.getMeta().getFileId().get());

        selected = selected.filterBy1(item -> mpfsInfoByResourceId.containsKeyTs(finalUid + ":" + item.getId()));

        if (sortByEtime) {
            selected = selected.sortedBy(t2 -> t2._1.getEtime().get());
        }

        return new SelectorResultPojo(
                uid, limit, from, to, newBeautyLimit, maxBeauty, matchLimit, text,

                selected.map((item, match) -> {
                    MpfsFileInfo mpfsFileInfo = mpfsInfoByResourceId.getTs(finalUid + ":" + item.getId());

                    return new PhotoInfo(
                            UrlUtils.removeParameter(mpfsFileInfo.getMeta().getPreview().get(), "size"),
                            mpfsFileInfo.getMeta().getFileUrl().get(),
                            match,
                            item.getBeautiful2()
                    );
                }).paginate(5).map(RowInfo::new)
        );
    }

    public static Instant parseDate(String dateStr) {
        SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
        simpleDateFormat.setTimeZone(TimeZone.getTimeZone("Europe/Moscow"));

        try {
            return new Instant(simpleDateFormat.parse(dateStr)).plus(Duration.standardHours(2));
        } catch (ParseException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Data
    @BenderBindAllFields
    @XmlRootElement(name = "content")
    static class SelectorResultPojo {
        public final String uid;
        public final int limit;

        public final String from;
        public final String to;

        public final double newBeautyLimit;
        public final double maxBeauty;
        public final long matchLimit;

        public final String text;

        public final ListF<RowInfo> rows;
    }

    @Data
    @BenderBindAllFields
    static class RowInfo {
        public final ListF<PhotoInfo> photos;
    }

    @Data
    @BenderBindAllFields
    static class PhotoInfo {
        public final String previewLink;
        public final String originalLink;

        public final long match;
        public final Option<Double> newBeauty;
    }
}
