package ru.yandex.chemodan.app.cvdemo2.admin;

import java.util.LinkedList;
import java.util.List;

import lombok.Data;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.random.Random2;

/**
 * @author tolmalev
 */
public class GridGenerator {

    private final GridGeneratorConfig config;

    public GridGenerator(GridGeneratorConfig config) {
        this.config = config;
    }

    public Grid generateGrid(List<PhotoInfo> photos) {
        Random2 R = new Random2(photos.size());

        ListF<Double> allBeauty = Cf.x(photos).flatMap(PhotoInfo::getBeauty);
        double averageBeauty = allBeauty.isEmpty() ? 0 : allBeauty.sum(Cf.Double) / allBeauty.size();

        BuildingGrid[] grids = new BuildingGrid[photos.size() + 1];
        grids[0] = new BuildingGrid(0.0, new BaseGrid(Cf.list()), Option.empty());
        int lastGoodGridIndex = 0;

        for (int i = 0; i < photos.size(); i++) {
            for (int j = Math.max(0, i - 10); j <= i; j++) {
                List<PhotoInfo> photosPart = photos.subList(j, i + 1);

                BuildingGrid prev = grids[j];
                if (prev == null) {
                    continue;
                }

                List<BaseGrid> baseGrids;
                if (config.cropAllTo.isPresent()) {
                    baseGrids = config.getBaseGrids(StringUtils.repeat(config.cropAllTo.get() + "", photosPart.size()));
                } else {
                    baseGrids = config.getBaseGrids(getTypesStr(photosPart));
                }

                if (baseGrids.isEmpty()) {
                    baseGrids = config.getBaseGrids(StringUtils.repeat("h", photosPart.size()));
                }

                for (BaseGrid grid : R.shuffle(baseGrids)) {
                    double newWeight = prev.weight + config.getWeightDiff(averageBeauty, prev, grid, photosPart);

                    if (grids[i + 1] == null || newWeight > grids[i + 1].weight) {
                        grids[i + 1] = new BuildingGrid(
                                newWeight,
                                grid,
                                Option.of(prev)
                        );
                    }
                }
            }

            if (grids[i + 1] == null && i - lastGoodGridIndex > 10 || i == photos.size() - 1) {
                BuildingGrid prev = grids[lastGoodGridIndex];
                List<PhotoInfo> photosPart = photos.subList(lastGoodGridIndex, i + 1);
                List<BaseGrid> baseGrids = config.getBaseGrids(StringUtils.repeat("h", photosPart.size()));

                for (BaseGrid grid : R.shuffle(baseGrids)) {
                    double newWeight = prev.weight + config.getWeightDiff(averageBeauty, prev, grid, photosPart);

                    if (grids[i + 1] == null || newWeight > grids[i + 1].weight) {
                        grids[i + 1] = new BuildingGrid(
                                newWeight,
                                grid,
                                Option.of(prev)
                        );
                    }
                }
            }

            if (grids[i + 1] != null) {
                lastGoodGridIndex = i + 1;
            }
        }

        if (grids[photos.size()] == null) {
            return null;
        }

        return buildGrid(grids[photos.size()]);
    }

    private static String getTypesStr(List<PhotoInfo> photos) {
        return Cf.x(photos)
                .map(p -> {
                    double aspect = p.getAspect();
                    if (0.9 <= aspect && aspect <= 1.1) {
                        return 's';
                    } else if (1.1 < aspect && aspect <= 2.0 / 1) {
                        return 'h';
                    } else if (4.0 / 2 < aspect) {
                        return '-';
                    } else {
                        return 'v';
                    }
                })
                .mkString("");
    }

    static Grid buildGrid(BuildingGrid builtGrid) {
        LinkedList<BaseGrid> grids = new LinkedList<>();
        while (builtGrid != null && builtGrid.totalH > 0) {
            grids.add(0, builtGrid.lastBaseGrid);
            builtGrid = builtGrid.prev.getOrNull();
        }
        return buildGrid(Cf.x(grids));
    }

    static Grid buildGrid(ListF<BaseGrid> baseGrids) {
        ListF<PhotoPosition> positions = Cf.arrayList();
        int H = 0;
        for (int i = 0; i < baseGrids.size(); i++) {
            BaseGrid grid = baseGrids.get(i);

            for (PhotoPosition position : grid.positions) {
                positions.add(new PhotoPosition(position.x, position.y + H, position.w, position.h));
            }

            H += grid.H;
        }

        return new Grid(positions);
    }

    @Data
    public static class GridGeneratorConfig {
        private final MapF<String, List<BaseGrid>> baseGridsByPhotoTypes;

        private final int normalPhotoSize;

        private final double beautyK;
        private final double normalSizeK;
        private final double cropK;
        private final double sizeDiffK;
        private final double photosInBlockK;
        private final double singlePhotoK;

        private final Option<Character> cropAllTo;

        public GridGeneratorConfig(MapF<String, List<BaseGrid>> baseGridsByPhotoTypes, int normalPhotoSize, double beautyK, double normalSizeK, double cropK, double sizeDiffK, double photosInBlockK, double singlePhotoK) {
            this.baseGridsByPhotoTypes = baseGridsByPhotoTypes;
            this.normalPhotoSize = normalPhotoSize;
            this.beautyK = beautyK;
            this.normalSizeK = normalSizeK;
            this.cropK = cropK;
            this.sizeDiffK = sizeDiffK;
            this.photosInBlockK = photosInBlockK;
            this.singlePhotoK = singlePhotoK;
            this.cropAllTo = Option.empty();
        }

        public GridGeneratorConfig(MapF<String, List<BaseGrid>> baseGridsByPhotoTypes, int normalPhotoSize, double beautyK, double normalSizeK, double cropK, double sizeDiffK, double photosInBlockK, double singlePhotoK, Option<Character> cropAllTo) {
            this.baseGridsByPhotoTypes = baseGridsByPhotoTypes;
            this.normalPhotoSize = normalPhotoSize;
            this.beautyK = beautyK;
            this.normalSizeK = normalSizeK;
            this.cropK = cropK;
            this.sizeDiffK = sizeDiffK;
            this.photosInBlockK = photosInBlockK;
            this.singlePhotoK = singlePhotoK;
            this.cropAllTo = cropAllTo;
        }

        public List<BaseGrid> getBaseGrids(String typesStr) {
            return baseGridsByPhotoTypes.getOrElse(typesStr, Cf.list());
        }

        public double getWeightDiff(double averageBeauty, BuildingGrid prev, BaseGrid grid, List<PhotoInfo> photos) {
            int photosCount = grid.positions.size();
            int differentSizesCount = Cf.x(grid.positions).map(PhotoPosition::size).unique().size();

            double beautyPart = Cf.x(photos)
                    .map(p -> p.beauty.getOrElse(averageBeauty))
                    .map(b -> Math.min(Math.max(b, -5.0), 5.0))
                    .zipWithIndex()
                    .map((b, i) -> (b - averageBeauty) * (grid.positions.get(i).size() - normalPhotoSize))
                    .sum(Cf.Double);

            double normalSizePart = Cf.x(photos)
                    .zipWithIndex()
                    .map((p, i) -> -Math.abs(grid.positions.get(i).size() - normalPhotoSize) * 1.0)
                    .sum(Cf.Double);

            double cropPart = Cf.x(photos)
                    .zipWithIndex()
                    .map((p, i) -> {
                        double photoAspect = p.getAspect();
                        double placeAspect = grid.positions.get(i).getAspect();

                        return 0.0;
                    })
                    .sum(Cf.Double);

            double sizeDiffPart = differentSizesCount * differentSizesCount;
            double photosInBlockPart = photosCount * photosCount;

            return -grid.H
                    + normalSizeK * normalSizePart
                    + beautyK * beautyPart
                    + sizeDiffK * sizeDiffPart
                    + cropK * cropPart
                    + photosInBlockK * photosInBlockPart
                    + (photosCount == 1 ? singlePhotoK : 0)
                    - (photosCount == 1 && prev.lastBaseGrid.positions.size() == 1 ? 100000 : 0)
                    ;
        }
    }

    @Data
    public static class PhotoInfo {
        public final int width;
        public final int height;
        public final Option<Double> beauty;

        public double getAspect() {
            return 1.0 * width / height;
        }
    }

    public static class Grid {
        public final int H;
        public final ListF<PhotoPosition> positions;

        public Grid(ListF<PhotoPosition> positions) {
            this.positions = positions;
            this.H = positions.map(pos -> pos.y + pos.h).max();
        }
    }

    @Data
    public static class PhotoPosition {
        public final int x;
        public final int y;
        public final int w;
        public final int h;

        public int size() {
            return w * h;
        }

        public double getAspect() {
            return 1.0 * w / h;
        }
    }

    public static class BaseGrid {
        public final int H;
        public final int W;

        public final List<PhotoPosition> positions;

        public BaseGrid(List<PhotoPosition> positions) {
            this.positions = positions;
            if (positions.isEmpty()) {
                this.H = this.W = 0;
            } else {
                this.H = Cf.x(positions).map(pos -> pos.y + pos.h).max();
                this.W = Cf.x(positions).map(pos -> pos.x + pos.w).max();
            }
        }

        public int size() {
            return W * H;
        }
    }

    static class BuildingGrid {
        public final int totalH;
        public final int totalW;

        public final double weight;

        public final BaseGrid lastBaseGrid;
        public final Option<BuildingGrid> prev;

        BuildingGrid(double weight, BaseGrid lastBaseGrid, Option<BuildingGrid> prev) {
            this.weight = weight;
            this.lastBaseGrid = lastBaseGrid;
            this.prev = prev;

            this.totalH = prev.map(p -> p.totalH).getOrElse(0) + lastBaseGrid.H;
            this.totalW = Math.max(prev.map(p -> p.totalW).getOrElse(0), lastBaseGrid.W);
        }
    }
}
