package ru.yandex.chemodan.app.cvdemo2.admin;

import lombok.Data;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.lentaloader.reminder.Cvi2tProcessor;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchClient;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.auth.YateamAuthUtils;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUser;
import ru.yandex.chemodan.zk.registries.staff.YandexStaffUserRegistry;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.commune.a3.action.parameter.bind.annotation.SpecialParam;
import ru.yandex.commune.admin.z.ZAction;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.XmlRootElement;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.InstantInterval;
import ru.yandex.misc.web.servlet.HttpServletRequestX;

@ActionContainer
public class SimilarityCvSelectorAdminPage {

    private final YandexStaffUserRegistry staffRegistry;
    private final DiskSearchClient diskSearchClient;
    private final MpfsClient mpfsClient;

    public SimilarityCvSelectorAdminPage(YandexStaffUserRegistry staffRegistry, DiskSearchClient diskSearchClient,
            MpfsClient mpfsClient)
    {
        this.staffRegistry = staffRegistry;
        this.diskSearchClient = diskSearchClient;
        this.mpfsClient = mpfsClient;
    }

    @ZAction(defaultAction = true, file = "similarity_cv_selector.xsl")
    @Path("/similarity-cv-selector")
    public SelectorResultPojo makeSelection(
            @SpecialParam HttpServletRequestX reqX,
            @RequestParam(value = "uid", required = false) String uid,
            @RequestParam(value = "limit", required = false) Option<Integer> limitO,
            @RequestParam(value = "from", required = false) String from,
            @RequestParam(value = "to", required = false) String to,

            @RequestParam(value = "matchLimit", required = false) Option<Long> matchLimitO,
            @RequestParam(value = "similarityBucketLimit", required = false) Option<Integer> similarityBucketLimitO
    )
    {
        uid = StringUtils.defaultIfEmpty(uid, "").trim();
        from = StringUtils.defaultIfEmpty(from, "2018-06-01").trim();
        to = StringUtils.defaultIfEmpty(to, "2018-09-01").trim();

        int limit = limitO.getOrElse(100);
        int similarityBucketLimit = similarityBucketLimitO.getOrElse(10);
        long matchLimit = matchLimitO.getOrElse(11200L);

        if (uid.isEmpty()) {
            Option<String> loginO = YateamAuthUtils.getLoginFromAttributeO(reqX);
            if (loginO.isPresent()) {
                String internalLogin = loginO.getOrThrow(() -> new RuntimeException("Not authorized"));

                YandexStaffUser user = staffRegistry.findByInternalLogin(internalLogin)
                        .getOrThrow(() -> new RuntimeException(internalLogin + ": логин на staff не привязан"));

                uid = Long.toString(user.externalUid.getUid());
            } else {
                uid = "208983871";
            }
        }
        ListF<DiskSearchFileInfo> searchFilesInfo = diskSearchClient.findPhotos(
                PassportUid.cons(Long.parseLong(uid)),
                new InstantInterval(CvSelectorAdminPage.parseDate(from), CvSelectorAdminPage.parseDate(to)),
                100000
        ).hitsArray.filter(info -> {
            byte[] i2tVector = info.geti2tVector();
            return i2tVector != null && i2tVector.length > 0;
        });
        MapF<DiskSearchFileInfo, ListF<SearchInfoWithSimilarity>> similarItems = findSimilarItems(searchFilesInfo,
                matchLimit, limit, similarityBucketLimit);
        MapF<String, MpfsFileInfo> mpfsData = Cf.hashMap();
        String finalUid = uid;
        mpfsData.putAll(similarItems.keys().paginate(50).flatMap(page -> {
            ListF<String> resourceIds = page.map(item -> finalUid + ":" + item.getId());
            return mpfsClient.bulkInfoByResourceIds(MpfsUser.of(finalUid), resourceIds, Cf.list("/disk", "/photounlim"));
        }).toMapMappingToKey(info -> finalUid + ":" + info.getMeta().getFileId().get()));
        mpfsData.putAll(similarItems.values().<SearchInfoWithSimilarity>flatten().paginate(50).flatMap(page -> {
            ListF<String> resourceIds = page.map(item -> finalUid + ":" + item.searchFileInfo.getId());
            return mpfsClient.bulkInfoByResourceIds(MpfsUser.of(finalUid), resourceIds, Cf.list("/disk", "/photounlim"));
        }).toMapMappingToKey(info -> finalUid + ":" + info.getMeta().getFileId().get()));
        ListF<RowInfo> rowInfos = Cf.arrayList();
        for (MapF.Entry<DiskSearchFileInfo, ListF<SearchInfoWithSimilarity>> entry : similarItems.entrySet()) {
            MpfsFileInfo sourceMpfsInfo = mpfsData.getTs(finalUid + ":" + entry.getKey().getId());
            PhotoInfo sourceInfo =
                    new PhotoInfo(UrlUtils.removeParameter(sourceMpfsInfo.getMeta().getPreview().get(), "size"),
                            sourceMpfsInfo.getMeta().getFileUrl().get(),
                    0L);
            ListF<PhotoInfo> photoInfos = entry.getValue().map(item -> {
                MpfsFileInfo mpfsFileInfo = mpfsData.getTs(finalUid + ":" + item.getSearchFileInfo().getId());
                return new PhotoInfo(UrlUtils.removeParameter(mpfsFileInfo.getMeta().getPreview().get(), "size"),
                        mpfsFileInfo.getMeta().getFileUrl().get(),
                        item.similarity);
            });
            rowInfos.add(new RowInfo(sourceInfo, photoInfos));
        }
        return new SelectorResultPojo(uid, limit, from, to, matchLimit, similarityBucketLimit, rowInfos);
    }

    private MapF<DiskSearchFileInfo, ListF<SearchInfoWithSimilarity>> findSimilarItems(ListF<DiskSearchFileInfo> items,
            long similarityThreshold, int countLimit, int bucketLimit)
    {
        MapF<DiskSearchFileInfo, ListF<SearchInfoWithSimilarity>> result = Cf.hashMap();
        for (DiskSearchFileInfo searchFileInfo : items) {
            for (DiskSearchFileInfo searchFileInfoToCompare : items) {
                if (searchFileInfo == searchFileInfoToCompare) {
                    continue;
                }
                long similarity = Cvi2tProcessor.dotProduct(searchFileInfo.geti2tVector(),
                        searchFileInfoToCompare.geti2tVector());
                if (similarity <= similarityThreshold) {
                    continue;
                }
                if (!result.getO(searchFileInfo).isPresent()) {
                    ListF<SearchInfoWithSimilarity> similarItems = Cf.arrayList();
                    result.put(searchFileInfo, similarItems);
                }
                ListF<SearchInfoWithSimilarity> similarItems = result.getTs(searchFileInfo);
                similarItems.add(new SearchInfoWithSimilarity(searchFileInfoToCompare, similarity));
                if (similarItems.size() == bucketLimit) {
                    break;
                }
            }
            if (result.size() == countLimit) {
                break;
            }
        }
        return result;
    }

    @Data
    @BenderBindAllFields
    @XmlRootElement(name = "content")
    static class SelectorResultPojo {
        public final String uid;
        public final int limit;

        public final String from;
        public final String to;

        public final long matchLimit;
        public final long bucketLimit;

        public final ListF<RowInfo> rows;
    }

    @Data
    @BenderBindAllFields
    static class RowInfo {
        public final PhotoInfo sourcePhoto;
        public final ListF<PhotoInfo> photos;
    }

    @Data
    @BenderBindAllFields
    static class PhotoInfo {
        public final String previewLink;
        public final String originalLink;

        public final long match;
    }

    @Data
    static class SearchInfoWithSimilarity {
        public final DiskSearchFileInfo searchFileInfo;
        public final long similarity;
    }
}
