package ru.yandex.chemodan.app.cvdemo2.core;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.cvdemo2.core.dao.CvBeautyDao;
import ru.yandex.chemodan.app.cvdemo2.core.dao.CvDataPojo;
import ru.yandex.chemodan.app.cvdemo2.worker.CalculateImageBeautyHighPriorityTask;
import ru.yandex.chemodan.app.cvdemo2.worker.CalculateImageBeautyTask;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchClient;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchFileInfo;
import ru.yandex.chemodan.app.lentaloader.reminder.DiskSearchResponse;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.InstantInterval;

/**
 * @author tolmalev
 */
public class CvBeautyManager {
    private static final Logger logger = LoggerFactory.getLogger(CvBeautyManager.class);

    public final DynamicProperty<Integer> prevDaysToIndex = new DynamicProperty<Integer>("cvdemo2-days-to-index", 7 * 8);

    private final CvBeautyDao beautyDao;
    private final MpfsClient mpfsClient;
    private final BazingaTaskManager bazingaTaskManager;

    private final MulcaClient mulcaClient;
    private final DiskSearchClient searchClient;
    private final CvApiClient cvApiClient;

    public CvBeautyManager(CvBeautyDao beautyDao, MpfsClient mpfsClient,
            BazingaTaskManager bazingaTaskManager,
            MulcaClient mulcaClient,
            DiskSearchClient searchClient,
            CvApiClient cvApiClient)
    {
        this.beautyDao = beautyDao;
        this.mpfsClient = mpfsClient;
        this.bazingaTaskManager = bazingaTaskManager;
        this.mulcaClient = mulcaClient;
        this.searchClient = searchClient;
        this.cvApiClient = cvApiClient;
    }

    public void reindexUser(String uid, boolean dropBeforeIndex) {
        if (dropBeforeIndex) {
            beautyDao.dropAllForUser(uid);
        }

        Instant now = Instant.now();
        PassportUid puid = PassportUid.cons(Long.parseLong(uid));
        for (int i = 0; i < prevDaysToIndex.get(); i++) {
            Instant end = now.minus(Duration.standardDays(i));
            Instant start = end.minus(Duration.standardDays(1));

            DiskSearchResponse photos = searchClient.findPhotos(puid, new InstantInterval(start, end), 1000);
            logger.info("Scheduling reindex for uid:{}, day:-{}, photos:{}", uid, i, photos.hitsArray.size());
            for (DiskSearchFileInfo info : photos.hitsArray) {
                scheduleImageRecalc(uid, uid + ":" + info.id, false);
            }
            logger.info("Scheduled");
        }
    }

    private void scheduleImageRecalc(String uid, String resourceId, boolean online) {
        if (online) {
            bazingaTaskManager.schedule(new CalculateImageBeautyHighPriorityTask(uid, resourceId));
        } else {
            bazingaTaskManager.schedule(new CalculateImageBeautyTask(uid, resourceId));
        }
    }

    public void recalculateSignatureAndStore(String uid, String resourceId) {
        String[] parts = resourceId.split(":");
        mpfsClient
                .getFileInfoOByFileId(MpfsUser.of(uid), parts[0], parts[1])
                .filter(info -> info.path.get().startsWith("/disk/") || info.path.get().startsWith("/photo"))
                .forEach(info -> {
                    try {
                        String stid = info.getMeta().getPmid().orElse(info.getMeta().getFileMid()).get();

                        byte[] bytes =
                                mulcaClient.download(MulcaId.fromSerializedString(stid)).readBytes();

                        String cvJson = cvApiClient.getCvJsonData(bytes);
                        long version = Long.parseLong(info.getMeta().getMetaJsonField("wh_version").get().getString());
                        beautyDao.insertOrUpdateCvData(uid, resourceId,
                                new Instant(info.times.etime.getOrElse(info.times.utime) * 1000),
                                version,
                                cvJson);
                    } catch (HttpException e) {
                        if (e.statusCodeIs(404)) {
                            return;
                        }
                        throw e;
                    }
                });
    }

    public ListF<CvDataPojo> getCvData(String uid, Instant from, Instant to) {
        return getCvData(uid, from, to, Option.empty());
    }

    public ListF<CvDataPojo> getCvData(String uid, Instant from, Instant to, Option<Integer> limit) {
        return beautyDao.getCvData(uid, from, to, limit);
    }

    public Tuple2List<CvDataPojo, MpfsFileInfo> zipWithFileInfo(ListF<CvDataPojo> data) {
        if (data.isEmpty()) {
            return Tuple2List.arrayList();
        }
        MpfsUser uid = MpfsUser.of(data.first().uid);

        MapF<String, MpfsFileInfo> infos = mpfsClient
                .bulkInfoByResourceIds(uid, data.map(d -> d.resourceId), Cf.list("/disk", "/photounlim"))
                .toMapMappingToKey(info -> info.getMeta().getResourceId().get().serialize());

        return data
                .filter(d -> infos.containsKeyTs(d.resourceId))
                .zipWith(d -> infos.getTs(d.resourceId));
    }
}
