package ru.yandex.chemodan.app.cvdemo2.worker;

import java.io.IOException;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.InputStreamEntity;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsStoreOperation;
import ru.yandex.chemodan.mpfs.MpfsStoreOperationContext;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockFullDescription;
import ru.yandex.chemodan.mpfs.lentablock.MpfsLentaBlockItemDescription;
import ru.yandex.chemodan.util.TimeUtils;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.io.exec.ExecResult;
import ru.yandex.misc.io.exec.ExecUtils;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.apache.v4.Abstract200ResponseHandler;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;

/**
 * @author tolmalev
 */
public class GenerateGifAnimationTask extends OnetimeTaskSupport<GenerateGifAnimationTask.Parameters> {

    private static final Logger logger = LoggerFactory.getLogger(GenerateGifAnimationTask.class);

    private final MpfsClient mpfsClient;
    private final MulcaClient mulcaClient;

    public GenerateGifAnimationTask(MpfsClient mpfsClient, MulcaClient mulcaClient) {
        super(Parameters.class);
        this.mpfsClient = mpfsClient;
        this.mulcaClient = mulcaClient;
    }

    public GenerateGifAnimationTask(String uid, Instant start, Instant end) {
        super(new Parameters(uid, start, end));

        mpfsClient = null;
        mulcaClient = null;
    }

    @Override
    public void execute(Parameters parameters, ExecutionContext context) throws Exception {

        mpfsClient
                .getFileInfoByUidAndPath(MpfsUser.of(parameters.uid), "/disk/Фотокамера", Cf.list("resource_id"))
                .getMeta()
                .getResourceId()
                .forEach(resourceId -> {
                    processBlock(parameters, resourceId.serialize());
                });

        mpfsClient
                .getFileInfoByUidAndPath(MpfsUser.of(parameters.uid), "/photounlim", Cf.list("resource_id"))
                .getMeta()
                .getResourceId()
                .forEach(resourceId -> {
                    processBlock(parameters, resourceId.serialize());
                });
    }

    private void processBlock(Parameters parameters, String resourceId) {
        Option<MpfsLentaBlockFullDescription> lentaBlockO =
                mpfsClient.getLentaBlockFilesData(MpfsUser.of(parameters.uid),
                        resourceId,
                        "image",
                        new MpfsUid(parameters.uid),
                        TimeUtils.unixTime(parameters.start),
                        TimeUtils.unixTimeTill(parameters.end),
                        10000,
                        "");

        if(!lentaBlockO.isPresent()) {
            logger.debug("No files found: {}", parameters);
            return;
        }

        MpfsLentaBlockFullDescription block = lentaBlockO.get();

        ListF<MpfsLentaBlockItemDescription> sortedResources = block.files
                .filter(i -> i.etime.isPresent())
                .filter(i -> i.preview.isPresent())
                .sortedBy(info -> info.etime.get());

        ListF<MpfsLentaBlockItemDescription> data = Cf.arrayList();
        if (sortedResources.isEmpty()) {
            return;
        }

        data.add(sortedResources.get(0));

        for (int i = 1; i < sortedResources.size(); i++) {
            Long prevEtime = sortedResources.get(i - 1).etime.get();
            Long etime = sortedResources.get(i).etime.get();

            if (etime - prevEtime > Duration.standardSeconds(3).getStandardSeconds()) {
                if (data.size() > 3) {
                    logger.debug("Found {} resources in {} - {}", data.size(), parameters.start, parameters.end);
                    generateGif(parameters.uid, data);
                }
                data.clear();
                data.add(sortedResources.get(i));
            } else {
                data.add(sortedResources.get(i));
            }
        }
    }

    private void generateGif(String uid, ListF<MpfsLentaBlockItemDescription> data) {
        ListF<String> resourceIds = data.map(i -> i.resourceId);

        ListF<MpfsFileInfo> fileInfos = mpfsClient.bulkInfoByResourceIds(
                MpfsUser.of(uid), resourceIds, Cf.list("/photounlim,/disk"));
        File2.withNewTempDir(dir -> {
            ListF<File2> paths = Cf.arrayList();

            int num = 1;
            for (MpfsFileInfo info : fileInfos) {
                File2 path = dir.child("in_" + num++ + "." + StringUtils.substringAfterLast(info.name.get(), "."));
                paths.add(path);

                mulcaClient
                        .download(MulcaId.fromSerializedString(info.getMeta().getPmid().get()))
                        .readTo(path);

                logger.debug("Downloaded file into: {}", path.getAbsolutePath());

                //ExecUtils.executeGrabbingOutput(Cf.list("convert", path.getAbsolutePath(), "-scale", "640x640", path.getAbsolutePath()));
            }

            File2 gifPath = dir.child("out.gif");

            File2 stabPrefix = dir.child("out_");
            ListF<String> stabCommand = Cf.arrayList("/Users/tolmalev/cv/a.out")
                    .plus(paths.map(File2::getAbsolutePath))
                    .plus(stabPrefix.getAbsolutePath());

            ExecResult stabResult = ExecUtils.executeGrabbingOutput(stabCommand);

            if (!stabResult.isSuccess()) {
                logger.warn("Failed to stabilize images for gif: {}", stabResult.getOutput());
                return;
            }

            ListF<String> stabPath = Cf
                    .range(1, paths.size() + 1)
                    .map(i -> stabPrefix.getAbsolutePath() + "_" + i + ".jpg")
                    .filter(path -> new File2(path).exists());

            if (stabPath.size() < 3) {
                logger.warn("Can't find enough photos: {}", stabPath.size());
                return;
            }

            ListF<String> command = Cf.arrayList("convert", "-loop", "0", "-delay", "20", "-scale", "640x640")
                    .plus(stabPath)
                    .plus(gifPath.getAbsolutePath());

            ExecResult result = ExecUtils.executeGrabbingOutput(command);

            if (result.isSuccess()) {
                String filename = Random2.R.nextAlnum(10) + ".gif";

                File2 file = new File2("/Users/tolmalev/Downloads").child(filename);
                file.write(gifPath.readBytes());

                MpfsStoreOperation store = mpfsClient
                        .store(MpfsStoreOperationContext
                                .builder()
                                .uid(MpfsUser.of(uid))
                                .path("/disk/Загрузки/" + filename)
                                .force(true)
                                .build(),
                                Tuple2List.tuple2List());

                HttpPut httpPut = new HttpPut(store.getUploadUrl().get());
                try {
                    httpPut.setEntity(new InputStreamEntity(gifPath.getInput(), gifPath.length()));
                } catch (IOException e) {
                    throw ExceptionUtils.translate(e);
                }

                ApacheHttpClientUtils.execute(httpPut, new Abstract200ResponseHandler<Void>() {
                    @Override
                    protected Void handle200Response(HttpResponse response)
                    {
                        logger.info("Uploaded gif file {}", filename);
                        return null;
                    }
                }, Timeout.seconds(60));

            } else {
                logger.warn("Failed to generate gif: {}", result.getOutput());
            }
        });
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardHours(1);
    }

    @Override
    public TaskQueueName queueName() {
        return CvDemo2TaskQueueName.CV_DEMO_CPU_INTENSIVE;
    }

    @BenderBindAllFields
    public static class Parameters extends DefaultObject {
        private final String uid;
        private final Instant start;
        private final Instant end;

        public Parameters(String uid, Instant start, Instant end) {
            this.uid = uid;
            this.start = start;
            this.end = end;
        }
    }
}
