package ru.yandex.chemodan.app.cvdemo2.worker;

import java.util.Optional;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.mpfs.MpfsClient;
import ru.yandex.chemodan.mpfs.MpfsFileInfo;
import ru.yandex.chemodan.mpfs.MpfsUser;
import ru.yandex.chemodan.util.encrypt.OpenSslAes256CbcCrypter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.bazinga.scheduler.TaskQueueName;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.inside.mulca.MulcaClient;
import ru.yandex.inside.mulca.MulcaId;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.DataSize;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.tables.TableWriterOptions;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class MakeCvExportTask extends OnetimeTaskSupport<MakeCvExportTask.Parameters> {
    private static final Logger logger = LoggerFactory.getLogger(MakeCvExportTask.class);

    private final OpenSslAes256CbcCrypter crypter;
    private final Yt yt;
    private final MpfsClient mpfsClient;
    private final MulcaClient mulcaClient;

    public MakeCvExportTask(Yt yt, MpfsClient mpfsClient, MulcaClient mulcaClient, OpenSslAes256CbcCrypter crypter) {
        super(Parameters.class);

        this.yt = yt;
        this.mpfsClient = mpfsClient;
        this.mulcaClient = mulcaClient;
        this.crypter = crypter;
    }

    public MakeCvExportTask(String ytPath, String dstPath, long fromIndex, long toIndex) {
        super(new Parameters(ytPath, dstPath, fromIndex, toIndex));

        this.yt = null;
        this.mpfsClient = null;
        this.mulcaClient = null;
        this.crypter = null;
    }

    @Override
    public int priority() {
        return 0;
    }

    @Override
    public Duration timeout() {
        return Duration.standardHours(1);
    }

    @Override
    protected void execute(Parameters parameters, ExecutionContext context) throws Exception {
        YPath dst = YPath.simple(parameters.dstPath);

//        if (yt.cypress().exists(dst)) {
//            yt.cypress().remove(dst);
//        }
//
//        yt.cypress().create(dst, CypressNodeType.TABLE, true, true,
//                Cf.map("dynamic", YTree.booleanNode(false))
//        );

        YPath path = YPath.simple(parameters.cvExportYtPath)
                .withRange(parameters.fromIndex, parameters.toIndex);

        ListF<Tuple2<String, ListF<YTreeMapNode>>> parts = yt.tables().read(path, YTableEntryTypes.YSON, it -> {
            ListF<YTreeMapNode> nodes = Cf.x(it).toList();

            return nodes
                    .groupBy(n -> n.getString("uid"))
                    .entries()
                    .flatMap(t2 -> t2._2.paginate(100).zipWith(node -> t2._1).invert());
        });

        ListF<FileLoadInfo> fileLoads = parts
                .map(t2 -> {
                    MapF<String, MpfsFileInfo> infos = mpfsClient
                            .bulkInfoByResourceIds(MpfsUser.of(t2._1), t2._2.map(n -> t2._1 + ":" + n.getString("file_id")),
                                    Cf.list("/disk", "/photounlim", "/trash"))
                            .toMapMappingToKey(m -> m.getMeta().getFileId().get());

                    return t2._2.flatMapO(n -> {
                        try {
                            FileLoadInfo i = new FileLoadInfo();

                            Option<MpfsFileInfo> infoO = infos.getO(i.file_id);

                            i.uid = t2._1;
                            i.file_id = n.getString("file_id");
                            i.stid = n.getString("stid");
                            i.preview_stid = infoO.flatMapO(inf -> inf.getMeta().getPmid());
                            i.upload_ycrid = n.getString("upload_ycrid");
                            i.upload_time = n.getString("upload_time");

                            i.trash_ycrid = Option.wrap(n.getStringO("upload_time"));
                            i.trash_time = Option.wrap(n.getStringO("trash_time"));

                            if (!i.trash_time.isPresent()
                                    && infoO.isPresent()
                                    && infoO.get().getMeta().getSource().isSome("trash"))
                            {
                                //hack trash time for files removed from trash
                                i.trash_time = Option.of(i.upload_time);
                            }

                            i.size = n.getLong("size");
                            i.etime = infoO.flatMapO(inf -> inf.times.etime);

                            return Option.of(i);
                        } catch (RuntimeException e) {
                            logger.error("Failed to parse node: {}", e);
                            return Option.empty();
                        }
                    });
                })
                .flatten();

        yt.tables().write(
                Optional.empty(),
                false,
                dst,
                YTableEntryTypes.YSON,
                fileLoads.iterator().filterMap(load -> {

                    if (!load.etime.isPresent() && load.size > ru.yandex.misc.dataSize.DataSize.fromMegaBytes(15).toBytes()) {
                        return Option.empty();
                    }

                    YTreeBuilder builder = YTree
                            .mapBuilder()
                            .key("uid").value(crypter.sign(load.uid))
                            .key("file_id").value(crypter.sign(load.file_id))
                            .key("stid").value(crypter.sign(load.stid))
                            .key("upload_ycrid").value(crypter.sign(load.upload_ycrid))
                            .key("upload_time").value(load.upload_time)
                            .key("size").value(load.size);

                    load.trash_time.forEach(v -> builder.key("trash_time").value(v));
                    load.trash_ycrid.forEach(v -> builder.key("trash_ycrid").value(crypter.sign(v)));
                    load.etime.forEach(v -> builder.key("etime").value(v));

                    String stid = load.stid;
                    if (load.etime.isPresent() && load.preview_stid.isPresent()) {
                        stid = load.preview_stid.get();
                    }


                    String finalStid = stid;
                    Either<Option<byte[]>, Throwable> fileContent = RetryUtils.retryE(logger, 10, () -> {
                        try {
                            return Option.of(mulcaClient.download(MulcaId.fromSerializedString(finalStid)).readBytes());
                        } catch (HttpException e) {
                            if (e.statusCodeIs(404)) {
                                return Option.empty();
                            }
                            throw e;
                        }
                    });

                    if (fileContent.isRight()) {
                        if (fileContent.getRight() instanceof HttpException) {
                            if (((HttpException) fileContent.getRight()).statusCodeIs(503)) {
                                return Option.empty();
                            }
                        }
                        ExceptionUtils.throwException(fileContent.getRight());
                    }

                    if (fileContent.getLeft().isPresent()) {
                        if (fileContent.getLeft().get().length > ru.yandex.misc.dataSize.DataSize.fromMegaBytes(15).toBytes()) {
                            return Option.empty();
                        }
                        builder.key("content").value(fileContent.getLeft().get());
                    } else {
                        return Option.empty();
                    }

                    return Option.of(builder.buildMap());
        }), new TableWriterOptions(DataSize.fromMegaBytes(100)));
    }

    @Override
    public TaskQueueName queueName() {
        return CvDemo2TaskQueueName.CV_DEMO_EXPORT;
    }

    @BenderBindAllFields
    static class Parameters {
        public final String cvExportYtPath;
        public final String dstPath;
        public final long fromIndex;
        public final long toIndex;

        Parameters(String cvExportYtPath, String dstPath, long fromIndex, long toIndex) {
            this.cvExportYtPath = cvExportYtPath;
            this.dstPath = dstPath;
            this.fromIndex = fromIndex;
            this.toIndex = toIndex;
        }
    }

    static class FileLoadInfo {
        public String uid;
        public String file_id;
        public String stid;
        public Option<String> preview_stid;
        public String upload_ycrid;
        public String upload_time;
        public Option<String> trash_ycrid;
        public Option<String> trash_time;

        public long size;
        public Option<Long> etime;
    }
}
