package ru.yandex.chemodan.app.datasyncadmin.idm;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContextSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.web.admin.AccessLevel;
import ru.yandex.chemodan.app.datasyncadmin.databases.AppDbs;
import ru.yandex.chemodan.app.datasyncadmin.databases.DatabasesRegistry;
import ru.yandex.chemodan.app.datasyncadmin.users.DbAccess;
import ru.yandex.chemodan.app.datasyncadmin.users.UserAccess;
import ru.yandex.chemodan.app.datasyncadmin.users.UsersRegistry;
import ru.yandex.chemodan.util.idm.IdmAllRolesResponse;
import ru.yandex.chemodan.util.idm.IdmError;
import ru.yandex.chemodan.util.idm.IdmErrorResponse;
import ru.yandex.chemodan.util.idm.IdmInfoResponse;
import ru.yandex.chemodan.util.idm.IdmOkResponse;
import ru.yandex.chemodan.util.idm.IdmResponse;
import ru.yandex.chemodan.util.idm.IdmRoutines;
import ru.yandex.chemodan.util.idm.MultilingualString;
import ru.yandex.chemodan.util.idm.RoleDescription;
import ru.yandex.chemodan.util.idm.RoleNode;
import ru.yandex.chemodan.util.idm.RoleSpecification;
import ru.yandex.chemodan.util.idm.SlugNode;

/**
 * @author yashunsky
 */
public class IdmRegistriesRoutines implements IdmRoutines {

    private final static String ROOT_KEY = "apps";
    private final static String DB_KEY = "dbs";

    public final static String ALL_DBS_MARKER = "*";

    private final static String SUPERUSER_KEY = "superuser";

    private final static RoleNode SUPERUSER = RoleNode.definitive(RoleDescription.simple(SUPERUSER_KEY));

    public final static RoleSpecification SUPERUSER_ROLE =
            new RoleSpecification(Tuple2List.fromPairs(ROOT_KEY, SUPERUSER_KEY));

    private final DatabasesRegistry databasesRegistry;
    private final AppSettingsRegistry appSettingsRegistry;
    private final ExternalDatabasesRegistry externalDatabasesRegistry;
    private final TypeSettingsRegistry typeSettingsRegistry;

    private final UsersRegistry usersRegistry;

    public IdmRegistriesRoutines(DatabasesRegistry databasesRegistry,
            AppSettingsRegistry appSettingsRegistry,
            ExternalDatabasesRegistry externalDatabasesRegistry,
            TypeSettingsRegistry typeSettingsRegistry,
            UsersRegistry usersRegistry)
    {
        this.databasesRegistry = databasesRegistry;
        this.appSettingsRegistry = appSettingsRegistry;
        this.externalDatabasesRegistry = externalDatabasesRegistry;
        this.typeSettingsRegistry = typeSettingsRegistry;
        this.usersRegistry = usersRegistry;
    }

    public ListF<AppDbs> getAllDatabases() {
        Tuple2List<String, String> custom = databasesRegistry.getAll()
                .map(a -> a.dbs.toTuple2List(d -> a.appId, d -> d)).reduceLeftO(Tuple2List::plus)
                .getOrElse(Tuple2List.fromPairs());

        Tuple2List<String, String> fromAppSettings = appSettingsRegistry.getAll()
                .filter(a -> a.databaseId.isPresent())
                .toTuple2List(a -> a.appName.getOrElse(".global"), a -> a.databaseId.get());

        Tuple2List<String, String> fromExternalDatabases =
                externalDatabasesRegistry.getAll().toTuple2List(d -> d.originalApp, d -> d.databaseId);

        Tuple2List<String, String> fromTypeSettings = typeSettingsRegistry.getAllTypeSettings()
                .map(TypeSettings::dbRef)
                .toTuple2List(DatabaseContextSource::dbAppId, DatabaseRef::databaseId);

        return custom
                .plus(fromAppSettings)
                .plus(fromExternalDatabases)
                .plus(fromTypeSettings)
                .map1(appName -> appName.equals(".@$GLOBAL") ? ".global" : appName)
                .groupBy1().mapValues(l -> l.unique().toList()).mapEntries(AppDbs::new);
    }

    @Override
    public IdmInfoResponse getInfo() {
        SlugNode dbSlug = SlugNode.definitiveDefaultWithSet(Cf.list(AccessLevel.values())
                .map(role -> RoleDescription.bilingual(role.value(), role.enDescription, role.ruDescription)));

        MapF<String, RoleNode> appsDbsRoles = getAllDatabases().toMap(
                appDbs -> appDbs.appId,
                appDbs -> {
                    String appId = appDbs.appId;
                    MapF<String, RoleNode> dbsRoles = appDbs.dbs.plus1(ALL_DBS_MARKER).toMap(
                            dbName -> dbName,
                            dbName -> RoleNode.intermediate(MultilingualString.singleValue(dbName), dbSlug));
                    SlugNode appSlug =
                            new SlugNode(DB_KEY, MultilingualString.bilingual("Databases", "Базы данных"), dbsRoles);
                    return RoleNode.intermediate(MultilingualString.singleValue(appId), appSlug);
                });

        SlugNode project = new SlugNode(ROOT_KEY, MultilingualString.bilingual("Applications", "Приложения"),
                appsDbsRoles.plus1(SUPERUSER_KEY, SUPERUSER));
        RoleNode info = RoleNode.root(project);
        return info.asIdmInfo();
    }

    private boolean isSuperuserSpec(RoleSpecification role) {
        return role.equals(SUPERUSER_ROLE);
    }

    private static Option<DbAccess> getDbAccess(RoleSpecification role) {
        ListF<String> values = role.getValues(Cf.list(ROOT_KEY, DB_KEY, SlugNode.ROLE_KEY));

        if (values.isEmpty()) {
            return Option.empty();
        }

        AccessLevel level;

        try {
            level = AccessLevel.valueOf(values.get(2).toUpperCase());
        } catch (IllegalArgumentException e) {
            return Option.empty();
        }

        return Option.of(new DbAccess(values.get(0), values.get(1), level));
    }

    public static RoleSpecification getRoleSpecification(DbAccess dbAccess) {
        return new RoleSpecification(Tuple2List.fromPairs(
                ROOT_KEY, dbAccess.appId, DB_KEY,
                dbAccess.dbId, SlugNode.ROLE_KEY,
                dbAccess.access.toDbValue()));
    }

    private boolean dbAccessible(String appId, String dbId) {
        Option<AppDbs> appDbsO = getAllDatabases().toMap(a -> a.appId, a -> a).getO(appId);

        if (dbId.equals(ALL_DBS_MARKER) && appDbsO.isPresent()) {
            return true;
        }

        return appDbsO.map(appDbs -> appDbs.dbs.containsTs(dbId)).getOrElse(false);
    }

    private boolean dbAccessible(DbAccess dbAccess) {
        return dbAccessible(dbAccess.appId, dbAccess.dbId);
    }

    @Override
    public IdmResponse addRole(String login, RoleSpecification role) {
        return usersRegistry.doSynchronized(() -> addRoleInner(login, role));
    }

    private IdmResponse addRoleInner(String login, RoleSpecification role) {
        UserAccess current = usersRegistry.getO(login).getOrElse(UserAccess.empty(login));
        UserAccess newAccess;

        if (isSuperuserSpec(role)) {
            newAccess = current.withSuperuser(true);
        } else {
            Option<DbAccess> accessO  = getDbAccess(role);

            if (accessO.isPresent() && dbAccessible(accessO.get())) {
                newAccess = current.withAccess(accessO.get());
            } else {
                return IdmErrorResponse.standard(IdmError.UNKNOWN_ROLE);
            }
        }

        usersRegistry.put(newAccess);
        return new IdmOkResponse();
    }

    @Override
    public IdmResponse removeRole(String login, RoleSpecification role, boolean fired) {
        return usersRegistry.doSynchronized(() -> removeRoleInner(login, role, fired));
    }

    private IdmResponse removeRoleInner(String login, RoleSpecification role, boolean fired) {
        Option<UserAccess> currentO = usersRegistry.getO(login);
        if (!currentO.isPresent()) {
            return new IdmOkResponse();
        }

        UserAccess current = currentO.get();
        UserAccess newAccess;

        if (isSuperuserSpec(role)) {
            newAccess = current.withSuperuser(false);
        } else {
            Option<DbAccess> accessO = getDbAccess(role);

            if (accessO.isPresent()) {
                newAccess = current.withoutAccess(accessO.get());
            } else {
                return IdmErrorResponse.standard(IdmError.UNKNOWN_ROLE);
            }
        }

        if (newAccess.equals(UserAccess.empty(login))) {
            usersRegistry.remove(login);
        } else {
            usersRegistry.put(newAccess);
        }
        return new IdmOkResponse();
    }

    @Override
    public IdmAllRolesResponse getAllRoles() {
        return new IdmAllRolesResponse(usersRegistry.getAll().map(UserAccess::getUserRoles));
    }
}
