package ru.yandex.chemodan.app.dataapi.api.data.field;


import org.joda.time.DateTime;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.Function;
import ru.yandex.misc.codec.FastBase64Coder;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;

/**
 * TODO: rename to DataFieldValue
 * @author tolmalev
 */
public class DataField extends DefaultObject {
    public final DataFieldType fieldType;
    public final Object value;

    public DataField(DataFieldType fieldType, Object value) {
        if (fieldType != DataFieldType.NULL) {
            Validate.isTrue(fieldType.valueClass.isInstance(value));
        } else {
            Validate.isNull(value);
        }
        switch (fieldType) {
            case INFINITY: Validate.isTrue(Double.isInfinite((Double) value)); break;
            case NEGATIVE_INFINITY: Validate.isTrue(Double.isInfinite((Double) value)); break;
            case NAN: Validate.isTrue(Double.isNaN((Double) value)); break;
            case LIST:
                for (Object o : (ListF) value) {
                    Validate.isTrue(o instanceof DataField);
                }
        }
        this.fieldType = fieldType;
        this.value = value;
    }

    public static DataField nan() {
        return new DataField(DataFieldType.NAN, Double.NaN);
    }

    public static DataField infinity() {
        return new DataField(DataFieldType.INFINITY, Double.POSITIVE_INFINITY);
    }

    public static DataField negativeInfinity() {
        return new DataField(DataFieldType.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY);
    }

    public static DataField bool(boolean value) {
        return new DataField(DataFieldType.BOOLEAN, value);
    }

    public static DataField integer(long value) {
        return new DataField(DataFieldType.INTEGER, value);
    }

    public static DataField decimal(double value) {
        return new DataField(DataFieldType.DECIMAL, value);
    }

    public static DataField string(String value) {
        return new DataField(DataFieldType.STRING, value);
    }

    public static DataField bytes(byte[] value) {
        return new DataField(DataFieldType.BYTES, value);
    }

    public static DataField nul() {
        return new DataField(DataFieldType.NULL, null);
    }

    public static DataField timestamp(Instant timestamp) {
        return new DataField(DataFieldType.TIMESTAMP, timestamp);
    }

    public static DataField dateTime(DateTime dateTime) {
        return new DataField(DataFieldType.DATETIME, dateTime);
    }

    public static DataField list(DataField... values) {
        return new DataField(DataFieldType.LIST, Cf.list(values));
    }

    public static DataField list(ListF<DataField> values) {
        return new DataField(DataFieldType.LIST, values);
    }

    public static DataField map(NamedDataField... fields) {
        return map(NamedDataField.toDataFieldMap(fields));
    }

    public static DataField map(MapF<String, DataField> map) {
        return new DataField(DataFieldType.MAP, map);
    }

    public String stringValue() {
        return (String) value;
    }

    public ListF<DataField> listValue() {
        return (ListF) value;
    }

    public Long integerValue() {
        return (Long) value;
    }

    public Instant timestampValue() {
        return (Instant) value;
    }

    public double decimalValue() {
        return (Double) value;
    }

    public MapF<String, DataField> mapValue() {
        return (MapF<String, DataField>) value;
    }

    public boolean booleanValue() {
        return (Boolean) value;
    }

    public byte[] bytesValue() {
        return (byte[]) value;
    }

    public DateTime dateTimeValue() {
        return (DateTime) value;
    }

    public static Function<String, DataField> stringF() {
        return DataField::string;
    }

    public static Function<Instant, DataField> timestampF() {
        return DataField::timestamp;
    }

    public static Function<DataField, String> stringValueF() {
        return DataField::stringValue;
    }

    public static Function<DataField, Instant> timestampValueF() {
        return DataField::timestampValue;
    }

    public static DataField cons(DataFieldType type, String value) {
        Object parsed = null;
        switch (type) {
            case DECIMAL: parsed = Cf.Double.parse(value); break;
            case INTEGER: parsed = Cf.Long.parse(value); break;
            case STRING: parsed = value; break;
            case BOOLEAN: parsed = Cf.Boolean.parse(value); break;
            case BYTES: parsed = FastBase64Coder.decode(value); break;
            case INFINITY: parsed = Double.POSITIVE_INFINITY; break;
            case NEGATIVE_INFINITY: parsed = Double.NEGATIVE_INFINITY; break;
            case NULL: parsed = null; break;
            case NAN: parsed = Double.NaN; break;
            case TIMESTAMP: parsed = new Instant(Cf.Long.parse(value)); break;
            case LIST: parsed = Cf.list(StringUtils.split(value, "\n")).map(stringF()); break;
            default: Validate.fail("Can't parse " + type + " " + value);
        }
        return new DataField(type, parsed);
    }

    public DataSize getSize() {
        if (fieldType.fixedSizeBytes.isPresent()) {
            return DataSize.fromBytes(fieldType.fixedSizeBytes.get());
        } else {
            DataSize size = DataSize.ZERO;
            switch (fieldType) {
                case STRING: {
                    size = getStringSizeInBytes((String) value);
                    break;
                }
                case BYTES: {
                    size = DataSize.fromBytes(((byte[]) value).length);
                    break;
                }
                case LIST: {
                    for (DataField field : (ListF<DataField>) value) {
                        size = size.plus(field.getSize());
                    }
                    break;
                }
                case MAP: {
                    MapF<String, DataField> map = (MapF<String, DataField>) value;
                    for(Tuple2<String, DataField> keyValue : map.entries()) {
                        size = size.plus(getStringSizeInBytes(keyValue.get1()));
                        size = size.plus(keyValue.get2().getSize());
                    }
                    break;
                }
                default:
                    throw new RuntimeException("Unknown field type " + fieldType);
            }

            return size;
        }
    }

    private DataSize getStringSizeInBytes(String stringValue) {
        return DataSize.fromBytes((stringValue).getBytes().length);
    }

    @Override
    public String toString() {
        switch (fieldType) {
            case BYTES:
                return StringUtils.format("{} ({})", fieldType, FastBase64Coder.encodeToString((byte[]) value));
            case INFINITY:
                return "inf";
            case NEGATIVE_INFINITY:
                return "-inf";
            case NULL:
                return "null";
            default:
                return value.toString();
        }
    }
}
