package ru.yandex.chemodan.app.dataapi.api.data.field;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.DataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.core.dao.support.DataApiRandomValueGenerator;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
public class DataFieldMarhsallerUnmarshallerTest {

    BenderMapper mapper = DataApiBenderUtils.mapper();

    @Test
    public void parseFromJsonTest() {
        String input =
                "{\n"
                + "    \"data\":{\n"
                + "        \"dateTimeExplicit\":{\n"
                + "            \"type\":\"datetime\",\n"
                + "            \"value\":\"2015-03-13T20:00:00.000+09:00\"\n"
                + "        },\n"
                + "        \"field1\":true,\n"
                + "        \"field2\":false,\n"
                + "        \"field3\":{\n"
                + "            \"type\":\"integer\",\n"
                + "            \"value\":1234\n"
                + "        },\n"
                + "        \"field4\":[\n"
                + "            123,\n"
                + "            null,\n"
                + "            {\n"
                + "                \"type\":\"integer\",\n"
                + "                \"value\":125\n"
                + "            },\n"
                + "            [\n"
                + "                123\n"
                + "            ]\n"
                + "        ],\n"
                + "        \"mapObject\":{\n"
                + "            \"type\":\"map\",\n"
                + "            \"value\":{\n"
                + "                \"intValue\":100,\n"
                + "                \"intValueExplicit\":{\n"
                + "                    \"type\":\"integer\",\n"
                + "                    \"value\":200\n"
                + "                },\n"
                + "                \"listValue\":[\n"
                + "                    300,\n"
                + "                    {\n"
                + "                        \"type\":\"integer\",\n"
                + "                        \"value\":400\n"
                + "                    }\n"
                + "                ],\n"
                + "                \"internalMap\":{\n"
                + "                    \"type\":\"map\",\n"
                + "                    \"value\":{\n"
                + "                        \"booleanValue\":true\n"
                + "                    }\n"
                + "                }\n"
                + "            }\n"
                + "        }\n"
                + "    }\n"
                + "}";

        TestClass actualObject = mapper.parseJson(TestClass.class, input);

        MapF<String, DataField> expectedDataMap = Cf.hashMap();
        expectedDataMap.put("dateTimeExplicit",
                DataField.dateTime(new DateTime(2015, 3, 13, 20, 0, DateTimeZone.forOffsetHours(9))));
        expectedDataMap.put("field1", DataField.bool(true));
        expectedDataMap.put("field2", DataField.bool(false));
        expectedDataMap.put("field3", DataField.integer(1234));
        DataField list = DataField.list(
                DataField.decimal(123), DataField.nul(), DataField.integer(125), DataField.list(DataField.decimal(123)));
        expectedDataMap.put("field4", list);

        MapF<String, DataField> mapObject = Cf.hashMap();
        mapObject.put("intValue", DataField.decimal(100));
        mapObject.put("intValueExplicit", DataField.integer(200));
        mapObject.put("listValue", DataField.list(DataField.decimal(300), DataField.integer(400)));

        MapF<String, DataField> internalMap = Cf.hashMap();
        internalMap.put("booleanValue", DataField.bool(true));
        mapObject.put("internalMap", DataField.map(internalMap));

        expectedDataMap.put("mapObject", DataField.map(mapObject));

        Assert.equals(new TestClass(expectedDataMap), actualObject);
    }

    @Test
    public void serializeToJsonTest() {
        String expectedResult =
                "{\n"
                + "    \"data\":{\n"
                + "        \"dateTime\":{\n"
                + "            \"type\":\"datetime\",\n"
                + "            \"value\":\"2015-03-13T20:00:00.000+09:00\"\n"
                + "        },\n"
                + "        \"internalMap\":{\n"
                + "            \"type\":\"map\",\n"
                + "            \"value\":{\n"
                + "                \"A\":\"A\",\n"
                + "                \"B\":\"B\"\n"
                + "            }\n"
                + "        },\n"
                + "        \"field11\":{\n"
                + "            \"type\":\"timestamp\",\n"
                + "            \"value\":1426534597163\n"
                + "        },\n"
                + "        \"field12\":null,\n"
                + "        \"field1\":true,\n"
                + "        \"field10\":{\n"
                + "            \"type\":\"binary\",\n"
                + "            \"value\":\"MjM0MTIzNTE1MzI=\"\n"
                + "        },\n"
                + "        \"field13\":[\n"
                + "            false,\n"
                + "            {\n"
                + "                \"type\":\"nan\"\n"
                + "            },\n"
                + "            {\n"
                + "                \"type\":\"integer\",\n"
                + "                \"value\":123123\n"
                + "            }\n"
                + "        ],\n"
                + "        \"field7\":123123.0,\n"
                + "        \"field6\":123123.123123,\n"
                + "        \"field9\":\"23412351532\",\n"
                + "        \"field8\":{\n"
                + "            \"type\":\"integer\",\n"
                + "            \"value\":123123\n"
                + "        },\n"
                + "        \"field3\":{\n"
                + "            \"type\":\"nan\"\n"
                + "        },\n"
                + "        \"field2\":false,\n"
                + "        \"field5\":{\n"
                + "            \"type\":\"-inf\"\n"
                + "        },\n"
                + "        \"field4\":{\n"
                + "            \"type\":\"+inf\"\n"
                + "        }\n"
                + "    }\n"
                + "}";

        String result = new String(mapper.serializeJson(new TestClass(getTestData())));
        Assert.equals(expectedResult.replaceAll(" |\\n", ""), result);
    }

    @Test
    public void randomParseSerialize() {
        TestClass test = new TestClass(randomTestData());
        Assert.equals(test, mapper.parseJson(TestClass.class, mapper.serializeJson(test)));
    }

    public static MapF<String, DataField> getTestData() {
        MapF<String, DataField> data = Cf.hashMap();

        data.put("dateTime", DataField.dateTime(new DateTime(2015, 3, 13, 20, 0, DateTimeZone.forOffsetHours(9))));
        data.put("field1", DataField.bool(true));
        data.put("field2", DataField.bool(false));
        data.put("field3", DataField.nan());
        data.put("field4", DataField.infinity());
        data.put("field5", DataField.negativeInfinity());
        data.put("field6", DataField.decimal(123123.123123));
        data.put("field7", DataField.decimal(123123));
        data.put("field8", DataField.integer(123123));
        data.put("field9", DataField.string("23412351532"));
        data.put("field10", DataField.bytes("23412351532".getBytes()));
        data.put("field11", DataField.timestamp(new Instant(1426534597163L)));
        data.put("field12", DataField.nul());
        data.put("field13", DataField.list(Cf.list(
                DataField.bool(false),
                DataField.nan(),
                DataField.integer(123123))));

        MapF<String, DataField> internalMap = Cf.hashMap();
        internalMap.put("A", DataField.string("A"));
        internalMap.put("B", DataField.string("B"));

        data.put("internalMap", DataField.map(internalMap));

        return data;
    }

    public static MapF<String, DataField> randomTestData() {
        final DataApiRandomValueGenerator R = new DataApiRandomValueGenerator();
        return Cf.x(DataFieldType.values())
                .toMap(type -> new Tuple2<>(Random2.R.nextString(10), R.createDataField(type)));
    }

    @Bendable
    private static class TestClass extends DefaultObject {
        @BenderPart
        public final MapF<String, DataField> data;

        private TestClass(MapF<String, DataField> data) {
            this.data = data;
        }
    }
}
