package ru.yandex.chemodan.app.dataapi.api.data.field;

import org.joda.time.DateTime;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.util.bender.JsonFieldLevelUnmarshaller;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.ParseResult;
import ru.yandex.misc.bender.parse.UnmarshallerContext;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.bender.serialize.MarshallerContext;
import ru.yandex.misc.bender.serialize.ToFieldMarshaller;
import ru.yandex.misc.codec.FastBase64Coder;
import ru.yandex.misc.xml.stream.XmlWriter;

/**
 * @author tolmalev
 */
public class DataFieldMarshallerUnmarshaller implements JsonFieldLevelUnmarshaller, ToFieldMarshaller {
    private static final String TYPE = "type";
    private static final String VALUE = "value";

    private static final String INTEGER_TYPE = "integer";
    private static final String BINARY_TYPE = "binary";
    private static final String TIMESTAMP_TYPE = "timestamp";
    private static final String DATETIME_TYPE = "datetime";
    private static final String NAN_TYPE = "nan";
    private static final String INFINITY_TYPE = "+inf";
    private static final String NEGATIVE_INFINITY_TYPE = "-inf";
    private static final String MAP_TYPE = "map";

    @Override
    public void writeXmlToField(XmlWriter writer, Object fieldValue, MarshallerContext context) {
        DataField field = (DataField) fieldValue;

        writer.textElement("type", field.fieldType.name().toLowerCase());
        if (field.fieldType != DataFieldType.LIST) {
            String textValue;
            switch (field.fieldType) {
                case NULL:
                    textValue = "null"; break;
                case BYTES:
                    textValue = new String(FastBase64Coder.encode((byte[]) field.value)); break;
                default:
                    textValue = field.value.toString();
            }
            writer.textElement("value", textValue);
        } else {
            for (DataField value : (ListF<DataField>) field.value) {
                writer.startElement("value");
                writeXmlToField(writer, value, context);
                writer.endElement();
            }
        }
    }

    public void writeJsonToField(BenderJsonWriter writer, Object fieldValue, MarshallerContext context) {
        writeJson(writer, fieldValue);
    }

    @Override
    public String getXmlFieldTextValue(Object o, MarshallerContext marshallerContext) {
        throw new RuntimeException("Not supported");
    }

    @Override
    public String getJsonFieldTextValue(Object o, MarshallerContext marshallerContext) {
        throw new RuntimeException("Not supported");
    }

    protected void writeJson(BenderJsonWriter json, Object o) {
        DataField field = (DataField) o;

        switch (field.fieldType) {
            case BOOLEAN:
                json.writeBoolean((Boolean) field.value);
                break;
            case NULL:
                json.writeNull();
                break;
            case DECIMAL:
                json.writeNumber(((Number) field.value).doubleValue());
                break;
            case STRING:
                json.writeString(field.value.toString());
                break;
            case INTEGER:
                json.writeObjectStart();
                json.writeFieldName(TYPE);
                json.writeString(INTEGER_TYPE);
                json.writeFieldName(VALUE);
                json.writeNumber(((Number) field.value).longValue());
                json.writeObjectEnd();
                break;
            case BYTES:
                json.writeObjectStart();
                json.writeFieldName(TYPE);
                json.writeString(BINARY_TYPE);
                json.writeFieldName(VALUE);
                json.writeString(new String(FastBase64Coder.encode((byte[]) field.value)));
                json.writeObjectEnd();
                break;
            case NAN:
                json.writeObjectStart();
                json.writeFieldName(TYPE);
                json.writeString(NAN_TYPE);
                json.writeObjectEnd();
                break;
            case INFINITY:
                json.writeObjectStart();
                json.writeFieldName(TYPE);
                json.writeString(INFINITY_TYPE);
                json.writeObjectEnd();
                break;
            case NEGATIVE_INFINITY:
                json.writeObjectStart();
                json.writeFieldName(TYPE);
                json.writeString(NEGATIVE_INFINITY_TYPE);
                json.writeObjectEnd();
                break;
            case TIMESTAMP:
                json.writeObjectStart();
                json.writeFieldName(TYPE);
                json.writeString(TIMESTAMP_TYPE);
                json.writeFieldName(VALUE);
                json.writeNumber(((Instant) field.value).getMillis());
                json.writeObjectEnd();
                break;
            case DATETIME:
                json.writeObjectStart();
                json.writeFieldName(TYPE);
                json.writeString(DATETIME_TYPE);
                json.writeFieldName(VALUE);
                json.writeString(field.value.toString());
                json.writeObjectEnd();
                break;
            case LIST:
                json.writeArrayStart();
                for (DataField element : (ListF<DataField>) field.value) {
                    writeJson(json, element);
                }
                json.writeArrayEnd();
                break;
            case MAP:
                json.writeObjectStart();
                json.writeFieldName(TYPE);
                json.writeString(MAP_TYPE);
                json.writeFieldName(VALUE);

                json.writeObjectStart();
                MapF<String, DataField> map = (MapF<String, DataField>) field.value;
                for (Tuple2<String, DataField> keyValue : map.entries()) {
                    json.writeFieldName(keyValue.get1());
                    writeJson(json, keyValue.get2());
                }
                json.writeObjectEnd();

                json.writeObjectEnd();
                break;

            default:
                throw new IllegalArgumentException("field.fieldType = " + field.fieldType);
        }
    }

    @Override
    public ParseResult<Object> parseJsonNode(BenderJsonNode json, final UnmarshallerContext context) {
        try {
            return ParseResult.result(parseDataField(json));
        } catch(UnknownTypeException e) {
            return e.type.isPresent()
                    ? ParseResult.failure("Unknown type " + e.type.get())
                    : ParseResult.failure("Type of field not given");
        }
    }

    public DataField parseDataField(BenderJsonNode json) {
        if (json.isNull()) {
            return DataField.nul();
        }

        if (json.isBoolean()) {
            return DataField.bool(json.getBooleanValueOrFalse());
        }

        if (json.isNumber()) {
            return DataField.decimal(
                    json.getNumberValueOrNull()
                            .doubleValue()
            );
        }

        if (json.isString()) {
            return new DataField(DataFieldType.STRING, json.getValueAsString());
        }

        if (json.isArray()) {
            return DataField.list(
                    json.getArrayElements()
                            .map(this::parseDataField)
            );
        }

        if (json.isObject()) {
            String type = json.getField(TYPE)
                    .getOrThrow(UnknownTypeException::new)
                    .getValueAsString();

            return parseNonDefaultType(type, json);
        }

        throw new UnknownTypeException();
    }

    private DataField parseNonDefaultType(String type, BenderJsonNode json) {
        switch (type) {
            case INTEGER_TYPE:
                return parseInteger(json);

            case NAN_TYPE:
                return DataField.nan();

            case INFINITY_TYPE:
                return DataField.infinity();

            case NEGATIVE_INFINITY_TYPE:
                return DataField.negativeInfinity();

            case BINARY_TYPE:
                return parseBinary(json);

            case TIMESTAMP_TYPE:
                return parseTimestamp(json);

            case DATETIME_TYPE:
                return parseDateTime(json);

            case MAP_TYPE:
                return parseMap(json);

            default:
                throw new UnknownTypeException(type);
        }
    }

    private DataField parseInteger(BenderJsonNode json) {
        String value = json.getField(VALUE).get().getValueAsString();
        return DataField.integer(Long.parseLong(value));
    }

    private DataField parseBinary(BenderJsonNode json) {
        String value = json.getField(VALUE).get().getValueAsString();
        return DataField.bytes(FastBase64Coder.decode(value));
    }

    private DataField parseTimestamp(BenderJsonNode json) {
        String value = json.getField(VALUE).get().getValueAsString();
        return DataField.timestamp(new Instant(Long.parseLong(value)));
    }

    private DataField parseDateTime(BenderJsonNode json) {
        String value = json.getField(VALUE).get().getValueAsString();
        return DataField.dateTime(DateTime.parse(value));
    }

    private DataField parseMap(BenderJsonNode json) {
        BenderJsonNode valueNode = json.getField(VALUE).get();
        MapF<String, DataField> map = Cf.hashMap();
        for (String name : valueNode.getFieldNames()) {
            DataField dataField = parseDataField(valueNode.getField(name).get());
            map.put(name, dataField);
        }
        return DataField.map(map);
    }

    private static class UnknownTypeException extends RuntimeException {
        final Option<String> type;

        private UnknownTypeException() {
            super("Type of field not given");
            this.type = Option.empty();
        }

        private UnknownTypeException(String type) {
            super("Unknown type " + type);
            this.type = Option.of(type);
        }
    }
}
