package ru.yandex.chemodan.app.dataapi.api.data.filter;

import java.util.function.Consumer;

import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.RecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.misc.db.q.SqlLimits;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class RecordsFilter {
    public static final RecordsFilter DEFAULT = new RecordsFilter(
            CollectionIdCondition.all(),
            RecordIdCondition.all(),
            RecordCondition.all(),
            RecordOrder.defaultOrder(),
            SqlLimits.all()
    );

    final CollectionIdCondition collectionIdCond;

    final RecordIdCondition recordIdCond;

    final RecordCondition recordCond;

    final RecordOrder recordOrder;

    final SqlLimits limits;

    private RecordsFilter(
            CollectionIdCondition collectionIdCond, RecordIdCondition recordIdCond, RecordCondition recordCond,
            RecordOrder recordOrder, SqlLimits limits)
    {
        this.collectionIdCond = collectionIdCond;
        this.recordIdCond = recordIdCond;
        this.recordCond = recordCond;
        this.recordOrder = recordOrder;
        this.limits = limits;
    }

    public boolean matches(DataRecord record) {
        return collectionIdCond.matches(record.getCollectionId())
                && recordIdCond.matches(record.getRecordId())
                && recordCond.matches(record);
    }

    public boolean hasAnyCondition() {
        return !collectionIdCond.isAll() || !recordIdCond.isAll() || !recordCond.isAll();
    }

    public boolean isDefaultOrdered() {
        return recordOrder == RecordOrder.defaultOrder();
    }

    public RecordsFilter withCollectionIdsO(Option<? extends CollectionF<String>> collectionIdsO) {
        return collectionIdsO.map(this::withCollectionIds)
                .getOrElse(this);
    }

    public RecordsFilter withCollectionIds(CollectionF<String> collectionIds) {
        return change(b -> b.collectionIdCond = CollectionIdCondition.inSet(collectionIds));
    }

    public RecordsFilter withColRef(CollectionRef colRef) {
        return withCollectionId(colRef.collectionId);
    }

    public RecordsFilter withRecordRef(RecordId recordId) {
        return withCollectionId(recordId.collectionId())
                .withRecordId(recordId.recordId());
    }

    public RecordsFilter withCollectionId(String collectionId) {
        return withCollectionIdCond(CollectionIdCondition.eq(collectionId));
    }

    public RecordsFilter withCollectionIdO(Option<String> collectionIdO) {
        return collectionIdO.map(this::withCollectionId)
                .getOrElse(this);
    }

    public RecordsFilter withCollectionIdCond(CollectionIdCondition collectionIdCond) {
        return change(b -> b.collectionIdCond = collectionIdCond);
    }

    public RecordsFilter withRecordId(String recordId) {
        return withRecordIdCond(RecordIdCondition.eq(recordId));
    }

    public RecordsFilter withRecordIdCond(RecordIdCondition recordIdCond) {
        return change(b -> b.recordIdCond = recordIdCond);
    }

    public RecordsFilter withDataCondO(Option<DataCondition> dataCondO) {
        return dataCondO.map(this::withDataCond)
                .getOrElse(this);
    }

    public RecordsFilter withDataCond(DataCondition recordCond) {
        return withRecordCond(recordCond);
    }

    public RecordsFilter withRecordCond(RecordCondition recordCond) {
        return change(b -> b.recordCond = recordCond);
    }

    public RecordsFilter withRecordOrderO(Option<? extends RecordOrder> recordOrderO) {
        return recordOrderO.map(this::withRecordOrder)
                .getOrElse(this);
    }

    public RecordsFilter withRecordOrder(RecordOrder recordOrder) {
        return change(b -> b.recordOrder = recordOrder);
    }

    public RecordsFilter withLimitsO(Option<SqlLimits> limitsO) {
        return limitsO.map(this::withLimits)
                .getOrElse(this);
    }

    public RecordsFilter withLimits(SqlLimits limits) {
        return change(b -> b.limits = limits);
    }

    public CollectionIdCondition getCollectionIdCond() {
        return collectionIdCond;
    }

    public RecordIdCondition getRecordIdCond() {
        return recordIdCond;
    }

    public RecordCondition getRecordCond() {
        return recordCond;
    }

    public RecordOrder getRecordOrder() {
        return recordOrder;
    }

    public SqlLimits limits() {
        return limits;
    }

    private RecordsFilter change(Consumer<Builder> builderF) {
        Builder builder = new Builder();
        builderF.accept(builder);
        return builder.build();
    }

    private class Builder {
        CollectionIdCondition collectionIdCond = RecordsFilter.this.collectionIdCond;

        RecordIdCondition recordIdCond = RecordsFilter.this.recordIdCond;

        RecordCondition recordCond = RecordsFilter.this.recordCond;

        RecordOrder recordOrder = RecordsFilter.this.recordOrder;

        SqlLimits limits = RecordsFilter.this.limits;

        RecordsFilter build() {
            return new RecordsFilter(collectionIdCond, recordIdCond, recordCond, recordOrder, limits);
        }
    }
}
