package ru.yandex.chemodan.app.dataapi.api.data.filter.condition;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlExpression;
import ru.yandex.misc.db.q.SqlQueryUtils;

/**
 * @author dbrylev
 */
public class DataColumn<T> extends RecordColumn<T, DataCondition> {

    private final Column column;
    private final Function<DataField, T> extractor;

    @SuppressWarnings("unchecked")
    private <C extends Comparable<C>> DataColumn(Column column, Function<DataField, T> extractor) {
        this(column, t -> (C) t, extractor);
    }

    private <C extends Comparable<C>> DataColumn(
            Column column, Function<T, C> converter, Function<DataField, T> extractor)
    {
        super(new ObjectFilter.ConvertingField<>(column, converter),
                r -> r.getFields().getO(column.name).map(extractor));

        this.column = column;
        this.extractor = extractor;
    }

    public static DataColumn<String> string(String field) {
        return new DataColumn<>(Column.string(field), DataField::stringValue);
    }

    public static DataColumn<Boolean> bool(String field) {
        return new DataColumn<>(Column.value(field, "boolean"), DataField::booleanValue);
    }

    public static DataColumn<Long> integer(String field) {
        return new DataColumn<>(Column.nestedValue(field, "bigint"), DataField::integerValue);
    }

    public static DataColumn<Double> decimal(String field) {
        return new DataColumn<>(Column.value(field, "double precision"), DataField::decimalValue);
    }

    public static DataColumn<Instant> timestamp(String field) {
        return new DataColumn<>(Column.nestedValue(field, "bigint"), Instant::getMillis, DataField::timestampValue);
    }

    @Override
    protected DataCondition condition(SqlCondition condition, Function<DataRecord, Option<Boolean>> matcher) {
        return new DataCondition(new RecordCondition.Matcher(
                condition, new RecordPredicate(matcher), DataFieldsCapture.of(getName())));
    }

    @Override
    protected ObjectFilter.ConvertingField<String, ?> stringField() {
        return new DataColumn<>(column.stringColumn(), extractor.andThen(Object::toString)).field;
    }

    public String getName() {
        return column.name;
    }

    protected static class Column extends SqlExpression {

        private final String name;
        private final Function<String, String> sqlF;
        private final Function<String, String> stringColumnSqlF;

        public Column(String name, Function<String, String> sqlF, Function<String, String> stringColumnSqlF) {
            this.name = name;
            this.sqlF = sqlF;
            this.stringColumnSqlF = stringColumnSqlF;
        }

        protected static Column cons(
                String name, Function2<String, String, String> fieldPath, Option<String> castType)
        {
            Function<String, String> fieldSqlF = fieldPath.bind2(SqlQueryUtils.quote(name));

            if (castType.isPresent()) {
                return new Column(name,
                        path -> "(" + fieldSqlF.apply(path) + ")::" + castType.get(),
                        path -> fieldSqlF.apply(path) + " COLLATE \"C\"");

            } else {
                return new Column(name, fieldSqlF, fieldSqlF);
            }
        }

        protected static Column string(String field) {
            return cons(field, (path, name) -> path + "->>" + name + " COLLATE \"C\"", Option.empty());
        }

        protected static Column value(String field, String sqlType) {
            return cons(field, (path, name) -> path + "->>" + name, Option.of(sqlType));
        }

        protected static Column nestedValue(String field, String sqlType) {
            return cons(field, (path, name) -> path + "->" + name + "->>'value'", Option.of(sqlType));
        }

        protected Column stringColumn() {
            return new Column(name, stringColumnSqlF, stringColumnSqlF);
        }

        @Override
        public ListF<Object> args() {
            return Cf.list();
        }

        @Override
        public String sql(Option<String> tableName) {
            return sqlF.apply(tableName.map(s -> s + ".").getOrElse("") + "jcontent->'data'");
        }
    }
}
