package ru.yandex.chemodan.app.dataapi.api.data.filter.condition;

import java.util.Objects;

import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByDataRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.util.postgres.PgSqlQueryUtils;
import ru.yandex.misc.db.q.ConditionUtils;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlExpression;
import ru.yandex.misc.regex.Pattern2;

/**
 * @author Dmitriy Amelin (lemeh)
 */
class ObjectFilter<T extends Comparable<T>> {
    private final Function1B<T> matcherF;

    private final SqlCondition sqlCondition;

    private ObjectFilter(Function1B<T> matcherF, SqlCondition sqlCondition) {
        this.matcherF = matcherF;
        this.sqlCondition = sqlCondition;
    }

    public boolean isAll() {
        return sqlCondition.isConstantTrue();
    }

    public <O> ListF<O> filter(ListF<O> objects, Function<O, T> mapperF) {
        return objects.filter(obj -> matches(obj, mapperF));
    }

    public <O> boolean matches(O object, Function<O, T> mapperF) {
        return matches(mapperF.apply(object));
    }

    public boolean matches(T inputValue) {
        return matcherF.apply(inputValue);
    }

    public SqlCondition getCondition() {
        return sqlCondition;
    }

    public Function1B<T> getMatcher() {
        return matcherF;
    }

    public ObjectFilter<T> not() {
        return new ObjectFilter<>(matcherF.notF(), sqlCondition.not());
    }

    public static <T extends Comparable<T>> Field<T> field(String name) {
        return new Field<>(name);
    }

    static class Field<T extends Comparable<T>> implements ConditionExpression<T, ObjectFilter<T>> {
        final SqlExpression column;

        private Field(String columnName) {
            this(ConditionUtils.column(columnName));
        }

        private Field(SqlExpression column) {
            this.column = column;
        }

        public ObjectFilter<T> all() {
            return cons(Function1B.trueF(), SqlCondition.trueCondition());
        }

        @Override
        public ObjectFilter<T> eq(T matchValue) {
            return cons(matchValue::equals, column.eq(matchValue));
        }

        @Override
        public ObjectFilter<T> ne(T mathValue) {
            return cons(((Function1B<T>) mathValue::equals).notF(), column.ne(mathValue));
        }

        @Override
        public ObjectFilter<T> gt(T matchValue) {
            return cons(inputValue -> inputValue.compareTo(matchValue) > 0, column.gt(matchValue));
        }

        @Override
        public ObjectFilter<T> ge(T matchValue) {
            return cons(inputValue -> inputValue.compareTo(matchValue) >= 0, column.ge(matchValue));
        }

        @Override
        public ObjectFilter<T> lt(T matchValue) {
            return cons(inputValue -> inputValue.compareTo(matchValue) < 0, column.lt(matchValue));
        }

        @Override
        public ObjectFilter<T> le(T matchValue) {
            return cons(inputValue -> inputValue.compareTo(matchValue) <= 0, column.le(matchValue));
        }

        @Override
        public ObjectFilter<T> inSet(CollectionF<? extends T> values) {
            return inSet(values.unique().uncheckedCast());
        }

        public ObjectFilter<T> inSet(SetF<T> values) {
            return cons(
                    values::containsTs,
                    values.isNotEmpty()
                            ? column.inSet(values)
                            : SqlCondition.falseCondition());
        }

        @Override
        public ObjectFilter<T> like(String pattern) {
            return cons(consSqlLikeMatcher(pattern), column.like(pattern));
        }

        @Override
        public ObjectFilter<T> between(T min, T max) {
            return cons(
                    value -> value.compareTo(min) >= 0 && value.compareTo(max) <= 0,
                    column.op("BETWEEN", PgSqlQueryUtils.between(min, max))
            );
        }

        @Override
        public ObjectFilter<T> isNull() {
            return cons(Objects::isNull, column.isNull());
        }

        @Override
        public ObjectFilter<T> isNotNull() {
            return cons(Objects::nonNull, column.isNotNull());
        }

        private ObjectFilter<T> cons(Function1B<T> matcherF, SqlCondition condition) {
            return new ObjectFilter<>(matcherF, condition);
        }
    }

    static class ConvertingField<F, T extends Comparable<T>>
            extends ConditionConvertingExpression<F, T, ObjectFilter<T>>
    {
        private final Function<F, T> converter;

        ConvertingField(String columnName, Function<F, T> converter) {
            this(ConditionUtils.column(columnName), converter);
        }

        ConvertingField(SqlExpression column, Function<F, T> converter) {
            super(new Field<T>(column));
            this.converter = converter;
        }

        @Override
        protected T convert(F value) {
            return converter.apply(value);
        }

        @SuppressWarnings("unchecked")
        public <D extends Comparable<D>> boolean matches(ObjectFilter<D> filter, F value) {
            return filter.matches((D) converter.apply(value));
        }

        public ByDataRecordOrder.Component orderComponent(Function<DataRecord, Option<F>> extractor, boolean asc) {
            return new ByDataRecordOrder.Component(getColumn(), rec -> extractor.apply(rec).map(converter), asc);
        }

        public SqlExpression getColumn() {
            return ((Field) expression).column;
        }
    }

    private static <T> Function1B<T> consSqlLikeMatcher(String sqlLike) {
        Pattern2 pattern = PgSqlQueryUtils.likePattern(sqlLike);

        return inputValue -> pattern.matches(inputValue.toString());
    }
}
