package ru.yandex.chemodan.app.dataapi.api.data.filter.condition;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.misc.db.q.ConstantCondition;
import ru.yandex.misc.db.q.SqlCondition;

/**
 * @author dbrylev
 */
public class RecordCondition {

    private final Matcher matcher;

    protected RecordCondition(Matcher matcher) {
        this.matcher = matcher;
    }

    public static RecordCondition all() {
        return new RecordCondition(Matcher.constant(true));
    }

    public static RecordCondition none() {
        return new RecordCondition(Matcher.constant(false));
    }

    public RecordCondition and(RecordCondition other) {
        return new RecordCondition(getMatcher().and(other.getMatcher()));
    }

    public RecordCondition or(RecordCondition other) {
        return new RecordCondition(getMatcher().or(other.getMatcher()));
    }

    public RecordCondition not() {
        return new RecordCondition(getMatcher().not());
    }

    public boolean matches(DataRecord record) {
        return matcher.matches(record);
    }

    public boolean isAll() {
        return getCondition().isConstantTrue();
    }

    public SqlCondition getCondition() {
        return matcher.condition;
    }

    protected Matcher getMatcher() {
        return matcher;
    }

    public SetF<String> getDataFields() {
        return matcher.fields.getFields();
    }

    protected static class Matcher {
        private final SqlCondition condition;
        private final RecordPredicate predicate;
        private final DataFieldsCapture fields;

        public Matcher(
                SqlCondition condition,
                RecordPredicate predicate,
                DataFieldsCapture fields)
        {
            this.condition = condition;
            this.predicate = predicate;
            this.fields = fields;
        }

        public static Matcher constant(boolean value) {
            return new Matcher(
                    new ConstantCondition(value), new RecordPredicate(rec -> Option.of(value)), DataFieldsCapture.empty());
        }

        public boolean matches(DataRecord record) {
            return predicate.matches(record);
        }

        public Matcher and(Matcher other) {
            return new Matcher(
                    condition.and(other.condition), predicate.and(other.predicate), fields.and(other.fields));
        }

        public Matcher or(Matcher other) {
            return new Matcher(
                    condition.or(other.condition), predicate.or(other.predicate), fields.or(other.fields));
        }

        public Matcher not() {
            return new Matcher(condition.not(), predicate.not(), fields.not());
        }
    }
}
