package ru.yandex.chemodan.app.dataapi.api.data.protobuf;

import java.util.Arrays;

import org.joda.time.DateTime;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.commune.protobuf5.annotation.ProtoMessage;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author tolmalev
 */
@ProtoMessage
@Bendable
@BenderBindAllFields
public class ProtobufDataFieldValue extends DefaultObject {
    @ProtoField(n = 1)
    private Option<Double> decimalValue = Option.empty();
    @ProtoField(n = 2)
    private Option<Long> integerValue = Option.empty();
    @ProtoField(n = 3)
    private Option<Boolean> booleanValue = Option.empty();
    @ProtoField(n = 4)
    private Option<String> stringValue = Option.empty();
    @ProtoField(n = 5)
    private Option<byte[]> bytesValue = Option.empty();
    @ProtoField(n = 6)
    private Option<Boolean> nullValue = Option.empty();
    @ProtoField(n = 7)
    private Option<Boolean> infinityValue = Option.empty();
    @ProtoField(n = 8)
    private Option<Boolean> negativeInfinityValue = Option.empty();
    @ProtoField(n = 9)
    private Option<Boolean> nanValue = Option.empty();
    @ProtoField(n = 10)
    private Option<Instant> timestampValue = Option.empty();
    @ProtoField(n = 11)
    private Option<Boolean> isList = Option.empty();
    @ProtoField(n = 12)
    private ListF<ProtobufDataFieldValue> listValue = Cf.list();
    @ProtoField(n = 13)
    private Option<Boolean> isMap = Option.empty();
    @ProtoField(n = 14)
    private ListF<ProtobufKeyValuePair> mapValue = Cf.arrayList();
    @ProtoField(n = 15)
    private Option<ProtobufDateTime> dateTimeValue = Option.empty();

    public ProtobufDataFieldValue(DataFieldType type, Object value) {
        switch (type) {
            case DECIMAL: decimalValue = Option.of((Double) value); break;
            case INTEGER: integerValue = Option.of((Long) value); break;
            case BOOLEAN: booleanValue = Option.of((Boolean) value); break;
            case STRING: stringValue = Option.of((String) value); break;
            case BYTES: bytesValue = Option.of((byte[]) value); break;
            case NULL: nullValue = Option.of(true); break;
            case INFINITY: infinityValue = Option.of(true); break;
            case NEGATIVE_INFINITY: negativeInfinityValue = Option.of(true); break;
            case NAN: nanValue = Option.of(true); break;
            case TIMESTAMP: timestampValue = Option.of((Instant) value); break;
            case DATETIME: {
                dateTimeValue = Option.of(new ProtobufDateTime((DateTime) value));
                break;
            }
            case LIST: {
                isList = Option.of(true);
                listValue = ((ListF<DataField>) value).map(ProtobufDataFieldValue::cons);
                break;
            }
            case MAP: {
                isMap = Option.of(true);
                MapF<String, DataField> map = (MapF<String, DataField>) value;
                mapValue = map.mapEntries((key, dataField) -> new ProtobufKeyValuePair(key, cons(dataField)));
                break;
            }
            default:
                throw new RuntimeException("Can't serialize " + type);
        }
    }

    public static ProtobufDataFieldValue cons(DataField field) {
        return new ProtobufDataFieldValue(field.fieldType, field.value);
    }

    public DataField toDataField() {
        if (decimalValue.isPresent()) {
            return DataField.decimal(decimalValue.get());
        } else if (integerValue.isPresent()) {
            return DataField.integer(integerValue.get());
        } else if (booleanValue.isPresent()) {
            return DataField.bool(booleanValue.get());
        } else if (stringValue.isPresent()) {
            return DataField.string(stringValue.get());
        } else if (bytesValue.isPresent()) {
            return DataField.bytes(bytesValue.get());
        } else if (nullValue.isSome(true)) {
            return DataField.nul();
        } else if (infinityValue.isSome(true)) {
            return DataField.infinity();
        } else if (negativeInfinityValue.isSome(true)) {
            return DataField.negativeInfinity();
        } else if (nanValue.isSome(true)) {
            return DataField.nan();
        } else if (timestampValue.isPresent()) {
            return DataField.timestamp(timestampValue.get());
        } else if (dateTimeValue.isPresent()) {
            return DataField.dateTime(dateTimeValue.get().toDateTime());
        } else if (isList.isSome(true)) {
            return DataField.list(listValue.map(ProtobufDataFieldValue::toDataField));
        } else if (isMap.isSome(true)) {
            return DataField.map(mapValue.toMap(kv -> new Tuple2<>(kv.key, kv.value.toDataField())));
        } else {
            throw new RuntimeException("Can't parse value");
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof ProtobufDataFieldValue)) {
            return false;
        }
        if (bytesValue.isPresent()) {
            ProtobufDataFieldValue protoObj = (ProtobufDataFieldValue) obj;
            return protoObj.bytesValue.isPresent()
                    && Arrays.equals(bytesValue.get(), protoObj.bytesValue.get());
        } else {
            return super.equals(obj);
        }
    }
}
