package ru.yandex.chemodan.app.dataapi.api.data.protobuf;

import org.joda.time.DateTime;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldMarhsallerUnmarshallerTest;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.field.NamedDataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasMarshallingTest;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.misc.io.ByteArrayInputStreamSource;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
public class ProtobufDataUtilsTest {
    private static final AppDatabaseRef DATABASE_REF = new AppDatabaseRef("app", "databaseId");

    private static final DatabaseHandle DATABASE_HANDLE = DATABASE_REF.consHandle("handle");

    private final DataApiPassportUserId user = new DataApiPassportUserId(101);

    @Test
    public void simpleParseSerializeRecord() {

        MapF<String, DataField> data = DataFieldMarhsallerUnmarshallerTest.randomTestData()
                .plus1("test-data", DataField.bytes(Random2.R.nextBytes(2048)));
        DataRecord record = createDataRecord(data);

        byte[] serialized = ProtobufDataUtils.serialize(record);
        Assert.equals(record, ProtobufDataUtils.parse(user, DATABASE_HANDLE, serialized));
        Assert.equals(
                Cf.list(record),
                ProtobufDataUtils.parse(user, DATABASE_HANDLE, Cf.list(serialized))
        );

        System.out.print("Serialized size: " + serialized.length);
    }

    @Test
    public void simpleParseSerializeDelta() {
        Delta delta = DeltasMarshallingTest.sampleDelta();

        byte[] serialized = ProtobufDataUtils.serialize(delta);
        Assert.equals(delta, ProtobufDataUtils.parseDelta(serialized));
        Assert.equals(delta, ProtobufDataUtils.parseDelta(new ByteArrayInputStreamSource(serialized)));

        System.out.print("Serialized size: " + serialized.length);
    }

    @Test
    public void serializeRecordWithMap() {
        byte[] data = serialize(
                NamedDataField.string("string1", "value1"),
                NamedDataField.integer("int", 202),
                NamedDataField.list("list", DataField.bool(true), DataField.bool(false)),
                NamedDataField.map("mapA",
                        NamedDataField.integer("int2", 303),
                        NamedDataField.list("list2",
                                DataField.integer(1), DataField.integer(2),
                                DataField.map(NamedDataField.bool("true", true))
                        ),
                        NamedDataField.map("mapBInMapA", NamedDataField.string("string2", "value2"))
                )
        );
        DataRecord record = ProtobufDataUtils.parse(user, DATABASE_HANDLE, data);

        @SuppressWarnings("unchecked")
        MapF<String, DataField> mapA = (MapF<String, DataField>) record.getData().getTs("mapA").value;
        @SuppressWarnings("unchecked")
        MapF<String, DataField> mapBInMapA = (MapF<String, DataField>) mapA.getTs("mapBInMapA").value;
        Assert.equals("value2", mapBInMapA.getTs("string2").value);
    }

    @Test
    public void serializeRecordWithDateTime() {
        DateTime dt = new DateTime(2015, 3, 13, 20, 0, org.joda.time.DateTimeZone.forOffsetHours(9));
        byte[] data = serialize(NamedDataField.dateTime("dateTime", dt));
        DataRecord record = ProtobufDataUtils.parse(user, DATABASE_HANDLE, data);
        Assert.equals(dt, record.getData().getTs("dateTime").dateTimeValue());
    }

    private byte[] serialize(NamedDataField... fields) {
        return ProtobufDataUtils.serialize(createDataRecord(fields));
    }

    private DataRecord createDataRecord(NamedDataField... fields) {
        return createDataRecord(new DataFields(fields));
    }

    private DataRecord createDataRecord(MapF<String, DataField> data) {
        return createDataRecord(new DataFields(data));
    }

    private DataRecord createDataRecord(DataFields fields) {
        DataRecordId recordId = new DataRecordId(DATABASE_HANDLE, "collectionId", "recordId");
        return new DataRecord(user, recordId, 10101, fields);
    }
}
