package ru.yandex.chemodan.app.dataapi.api.data.snapshot;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecords;
import ru.yandex.chemodan.app.dataapi.api.data.record.ModifiableDataRecords;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.SpecialDatabases;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaValidationMode;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class PatchableSnapshot extends DefaultObject {
    public final DeltaValidationMode validationMode;

    public final Snapshot initialSnapshot;

    public final ModifiableDataRecords patchedRecords;

    public final ListF<Delta> deltas;

    public final ListF<Delta> extraDeltas;

    public PatchableSnapshot(Snapshot snapshot) {
        this(snapshot, Option.empty());
    }

    public PatchableSnapshot(Snapshot snapshot, Option<ByIdRecordOrder> order) {
        this(snapshot, SpecialDatabases.isHackDeleteAllowed(snapshot.database.dbRef())
                ? DeltaValidationMode.IGNORE_NON_EXISTENT_DELETION
                : DeltaValidationMode.STRICT,
                snapshot.records.toModifiable(order));
    }

    private PatchableSnapshot(
            Snapshot initialSnapshot, DeltaValidationMode validationMode, ModifiableDataRecords patchedRecords)
    {
        this(initialSnapshot, validationMode, patchedRecords, Cf.arrayList(), Cf.arrayList());
    }

    private PatchableSnapshot(
            Snapshot initialSnapshot, DeltaValidationMode validationMode, ModifiableDataRecords patchedRecords,
            ListF<Delta> deltas, ListF<Delta> extraDeltas)
    {
        this.initialSnapshot = initialSnapshot;
        this.validationMode = validationMode;
        this.patchedRecords = patchedRecords;
        this.deltas = deltas;
        this.extraDeltas = extraDeltas;
    }

    public PatchableSnapshot withValidationMode(DeltaValidationMode mode) {
        return new PatchableSnapshot(initialSnapshot, mode, patchedRecords);
    }

    public PatchableSnapshot patch(Delta delta) {
        return patch(Cf.list(delta));
    }

    public PatchableSnapshot patch(CollectionF<Delta> deltas) {
        // First of all, we apply changes in memory, and only than in DB
        // This helps to deal with complex updates like update - delete - create - update - ...
        deltas.forEach(this::apply);

        return this;
    }

    public DatabaseChange toDatabaseChange() {
        return new DatabaseChange(initialSnapshot, unmodifiable(), deltas.plus(extraDeltas));
    }

    private void apply(Delta delta) {
        delta.apply(this);
        (delta.extra ? extraDeltas : deltas).add(delta.withSequentialRevs(currentRev()));
    }

    public Snapshot unmodifiable() {
        return new Snapshot(consPatchedDatabase(), patchedRecords.unmodifiable());
    }

    private Database consPatchedDatabase() {
        return database()
                .withSizeInc(patchedRecords.calcDataSize() - initialRecords().calcDataSize())
                .withIncRecordsCount(patchedRecords.recordCount() - initialRecords().recordCount())
                .withIncRev(revInc());
    }

    public Database database() {
        return initialSnapshot.database;
    }

    public long nextRev() {
        return currentRev() + 1;
    }

    private long currentRev() {
        return database().rev + revInc();
    }

    private int revInc() {
        return deltas.size() + extraDeltas.size();
    }

    private DataRecords initialRecords() {
        return initialSnapshot.records;
    }
}
