package ru.yandex.chemodan.app.dataapi.api.datasource;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.junit.After;
import org.junit.AssumptionViolatedException;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.internal.runners.statements.ExpectException;
import org.junit.rules.TestRule;
import org.junit.runner.Description;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.model.Statement;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordCondition;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.PatchableSnapshotTest;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotPojoRow;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseExistsException;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaValidationException;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManagerImpl;
import ru.yandex.chemodan.app.dataapi.core.xiva.DataApiXivaPushSender;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.dataapi.web.DeltasGoneException;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.devtools.test.YaTest;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@RunWith(Parameterized.class)
@ContextConfiguration(classes = SessionProviderTestContextConfiguration.class)
public class DataSourceSessionImplTest extends DataApiTestSupport {
    private static final Logger logger = LoggerFactory.getLogger(DataSourceSessionImplTest.class);

    @Rule
    public final TestRule customRule = new CustomRule();

    private final DataSourceType type;

    @Autowired
    private DataApiManager dataApiManager;

    @Autowired
    private DataApiXivaPushSender xivaPushSender;

    private MapF<DataSourceType, DataSourceProvider> dataSourceProviders;

    private DataApiUserId testUser;

    private Option<AppDatabaseRef> databaseRef = Option.empty();

    public DataSourceSessionImplTest(DataSourceType type) {
        this.type = type;
    }

    @Parameterized.Parameters(name = "{0}")
    public static Iterable<Object[]> data() {
        if (YaTest.insideYaTest) {
            return Cf.list(DataSourceType.DISK).map(t -> new Object[]{t});
        } else {
            return Cf.list(DataSourceType.values())
                    .map(t -> new Object[]{t});
        }
    }

    protected DataSourceProvider getSessionProvider() {
        return dataSourceProviders.getTs(type);
    }

    @Before
    public void setUp() {
        this.testUser = getSessionProvider().nextUser();
        userInitializer.initUserForTests(this.testUser, false);
        cleanup();
        Mockito.reset(recordsDao);
        setUpInternal();
    }

    /**
     * Override in subclasses
     */
    protected void setUpInternal() {

    }

    @After
    public void tearDown() {
        databaseRef = Option.empty();
    }

    @Test
    public void testGetOrCreateDatabase() {
        Database database1 = openSession().getOrCreateDatabase();
        Assert.isTrue(database1.isNew);

        Database database2 = openSession().getOrCreateDatabase();
        Assert.isFalse(database2.isNew);
        Assert.equals(database1.asExisting(), database2);
    }

    @Test(expected = DatabaseExistsException.class)
    public void testCreateDatabase() {
        Database database = openSession().createDatabase();
        Assert.isTrue(database.isNew);

        openSession().createDatabase();
    }

    @Test
    public void testGetDatabaseO() {
        Assert.none(openSession().getDatabaseO());
        openSession().createDatabase();
        Assert.some(openSession().getDatabaseO());
    }

    @Test
    @ExceptionIfTrait(trait = DataSourceTrait.NO_DB_REMOVAL, exception = UnsupportedOperationException.class)
    public void testMarkDatabaseDeleted() {
        testDatabaseDeleted(DatabaseDeletionMode.MARK_DELETED, 2);
    }

    @Test
    @ExceptionIfTrait(trait = DataSourceTrait.NO_DB_REMOVAL, exception = UnsupportedOperationException.class)
    public void testRemoveDatabaseCompletely() {
        testDatabaseDeleted(DatabaseDeletionMode.REMOVE_COMPLETELY, 0);
    }

    private void testDatabaseDeleted(DatabaseDeletionMode deletionMode, int expectedRecordCount) {
        Database database = createDatabaseWithRecords().patchedDatabase();
        openSession().deleteDatabase(deletionMode);
        int actualRecordCount = openSessionByHandle(database.dbHandle)
                .getDataRecordsCount(RecordsFilter.DEFAULT);
        Assert.equals(expectedRecordCount, actualRecordCount);
    }

    @Test
    @ExceptionIfTrait(trait = DataSourceTrait.NO_DB_DESCRIPTION, exception = UnsupportedOperationException.class)
    public void testSetDescription() {
        String description = "test database";
        openSession().createDatabase();
        openSession().setDatabaseDescription(Option.of(description));
        Assert.equals(description, openSession().getDatabase().meta.description.get());
    }

    @Test
    @ExceptionIfTrait(trait = DataSourceTrait.NEVER_KEEP_DELTAS, exception = NotFoundException.class)
    public void testGetDelta() {
        DatabaseChange databaseChange = createDatabase(
                Delta.fromNewRecords(getSessionProvider().nextRecord())
        );
        Delta expectedDelta = databaseChange.deltas.first();
        Delta actualDelta = openSession().getDelta(expectedDelta.rev.get());
        Assert.equals(expectedDelta, actualDelta);
    }

    @Test
    @ExceptionIfTrait(trait = DataSourceTrait.NEVER_KEEP_DELTAS, exception = DeltasGoneException.class)
    public void testListDeltas() {
        DatabaseChange databaseChange = createDatabase(
                Delta.fromNewRecords(getSessionProvider().nextRecord()),
                Delta.fromNewRecords(getSessionProvider().nextRecord())
        );
        ListF<Delta> actualDeltas = openSession().listDeltas(0, 100);
        Assert.equals(databaseChange.deltas, actualDeltas);
    }

    @Test
    public void testGetSnapshotO() {
        Assert.none(openSession().getSnapshotO(RecordsFilter.DEFAULT));

        ListF<SimpleDataRecord> records = generateRecords(3);
        Database database = createDatabaseWithRecords(records).patchedDatabase();
        Snapshot snapshot = openSession().getSnapshotO(RecordsFilter.DEFAULT.withLimits(SqlLimits.first(3))).get();

        Database expectedDatabase = database.asExisting();
        if (dataSourceHasTrait(DataSourceTrait.NO_DB_SIZE)) {
            expectedDatabase = expectedDatabase.withSize(DataSize.ZERO);
        }

        expectedDatabase = expectedDatabase.withModificationTime(snapshot.database.meta.modificationTime);

        if (yaMoneyIsUnderTest()) {
            // ignore database revision
            expectedDatabase = expectedDatabase.withRev(snapshot.database.rev);
        }

        Assert.equals(expectedDatabase, snapshot.database);
        Assert.equals(unifyRecord(records), unifyRecord(snapshot.records.toSimpleRecords()));
    }

    @Autowired
    private DataRecordsJdbcDao recordsDao;

    @Test
    public void testGetDataRecordsO() {
        Assert.none(openSession().getDataRecordsO(RecordsFilter.DEFAULT));

        ListF<SimpleDataRecord> records = generateRecords(3);
        createDatabaseWithRecords(records);

        ListF<SimpleDataRecord> actualRecords = openSession()
                .getDataRecords(RecordsFilter.DEFAULT)
                .map(SimpleDataRecord::new);

        Assert.equals(unifyRecord(records), unifyRecord(actualRecords));

        if (type != DataSourceType.DISK) {
            return;
        }

        Mockito
                .inOrder(recordsDao)
                .verify(recordsDao, Mockito.calls(1))
                .find(
                        Mockito.any(),
                        Mockito.any(DatabaseRef.class),
                        Mockito.any(), Mockito.any(), Mockito.any(), Mockito.any(), Mockito.any()
                );

        Mockito
                .inOrder(recordsDao)
                .verify(recordsDao, Mockito.never())
                .find(
                        Mockito.any(),
                        Mockito.any(DatabaseHandle.class),
                        Mockito.any(), Mockito.any(), Mockito.any(), Mockito.any(), Mockito.any()
                );
    }

    @Test
    public void testGetDataRecordCount() {
        Assert.equals(0, openSession().getDataRecordsCount(RecordsFilter.DEFAULT));

        createDatabaseWithRecords(generateRecords(3));
        Assert.equals(3, openSession().getDataRecordsCount(RecordsFilter.DEFAULT));
    }

    @Test
    @ExceptionIfNoTrait(trait = DataSourceTrait.EXTERNAL_CHANGE_ALLOWED,
            exception = UnsupportedOperationException.class)
    public void testSetRevision() {
        xivaPushSender.doSendPush(Mockito.any(), Mockito.any(), Mockito.anyLong());
        openSession().onDatabaseUpdate(1);
    }

    @Test
    public void testApplyDelta() {
        createDatabaseWithRecords();

        SetF<DataRecordId> recordIds = openSession().getSnapshotO(RecordsFilter.DEFAULT).get().recordIds();
        Assert.notEmpty(recordIds);

        Delta delta = new Delta(recordIds.map(RecordChange::delete));

        retryIfYaMoneyDeltasGone(() -> {
            openSession().applyDeltas(Cf.list(delta),
                    DataApiManagerImpl.RevisionChecker.ifMatch(RecordCondition.all()));

            Assert.isEmpty(openSession()
                    .getSnapshotO(RecordsFilter.DEFAULT)
                    .get()
                    .records()
            );
        });

        retryIfYaMoneyDeltasGone(() -> Assert.assertThrows(() -> openSession()
                        .applyDeltas(Cf.list(delta),
                                DataApiManagerImpl.RevisionChecker.ifMatch(RecordCondition.all())),
                DeltaValidationException.class));
    }

    private void retryIfYaMoneyDeltasGone(Runnable runnable) {
        if (!yaMoneyIsUnderTest()) {
            runnable.run();
            return;
        }

        RetryUtils.retryOrThrow(logger, 5,
                Function0V.wrap(runnable)
                        .asFunction0ReturnNull(),
                ex -> !(ex instanceof DeltasGoneException)
        );
    }

    @Test
    @SkipForType(DataSourceType.YA_MONEY)
    public void testApplyDeltas() {
        for (String caseName : Cf.range(1, 7).map(caseN -> "case" + caseN)) {
            setUp();

            ListF<SnapshotPojoRow> rowsStart = PatchableSnapshotTest.readSnapshot(caseName, "input.json");
            Delta delta = PatchableSnapshotTest.readDelta(caseName);
            ListF<SnapshotPojoRow> rowsEnd = PatchableSnapshotTest.readSnapshot(caseName, "output.json")
                    .map(row -> row.withRev(delta.changes.isNotEmpty() ? 1 : 0));

            ListF<SimpleDataRecord> recordsStart =
                    rowsStart.map(r -> new SimpleDataRecord(r.collectionId, r.recordId, new DataFields(r.getData())));

            DatabaseChange change = createDatabaseWithRecords(recordsStart);

            openSession().applyDeltas(
                    Cf.list(delta),
                    DataApiManagerImpl.RevisionChecker.cons(RevisionCheckMode.WHOLE_DATABASE, change.patchedDatabase().rev)
            );

            ListF<SnapshotPojoRow> realRecords = openSession()
                    .getSnapshotO(RecordsFilter.DEFAULT).get().records.records().unique().map(DataRecord::toSnapshotPojoRow);

            Assert.equals(
                    rowsEnd.map(r -> new Tuple3<>(r.collectionId, r.recordId, r.getData())).unique(),
                    realRecords.map(r -> new Tuple3<>(r.collectionId, r.recordId, r.getData())).unique()
            );

            tearDown();
        }
    }

    @Test(expected = DeltaValidationException.class)
    public void throwDeltaValidationWhenInsertingSameRecordTwice() {
        openSession().createDatabase();
        Delta insertDelta = Delta.fromNewRecords(getSessionProvider().nextRecord());
        openSession().applyDeltas(Cf.list(insertDelta),
                DataApiManagerImpl.RevisionChecker.ifMatch(RecordCondition.all()));
        openSession().applyDeltas(Cf.list(insertDelta),
                DataApiManagerImpl.RevisionChecker.ifMatch(RecordCondition.all()));
    }

    private DatabaseChange createDatabaseWithRecords() {
        return createDatabaseWithRecords(generateRecords(2));
    }

    private DatabaseChange createDatabaseWithRecords(ListF<SimpleDataRecord> records) {
        return createDatabase(Delta.fromNewRecords(records));
    }

    private DatabaseChange createDatabase(Delta... deltas) {
        return createDatabase(Cf.list(deltas));
    }

    private DatabaseChange createDatabase(CollectionF<Delta> deltas) {
        DatabaseChange change = Snapshot.empty(openSession().createDatabase())
                .toPatchable()
                .patch(deltas)
                .toDatabaseChange();
        openSession().save(change);
        return change;
    }

    private ListF<SimpleDataRecord> generateRecords(int recordCount) {
        return Cf.x(
                Stream.generate(getSessionProvider()::nextRecord)
                        .limit(recordCount)
                        .collect(Collectors.toList())
        );
    }

    private DataApiManagerImpl.TransactionalManagerSession openSession() {
        return openSession(UserDatabaseSpec.fromUserAndAlias(testUser, getDatabaseAlias()));
    }

    private DataApiManagerImpl.TransactionalManagerSession openSessionByHandle(DatabaseHandle handle) {
        return openSession(UserDatabaseSpec.fromUserAndHandle(testUser, handle));
    }

    private DataApiManagerImpl.TransactionalManagerSession openSession(UserDatabaseSpec databaseSpec) {
        return dataApiManager.openSession(getSessionProvider().dataSource(), databaseSpec);
    }

    private boolean dataSourceHasTrait(DataSourceTrait trait) {
        return type.hasTrait(trait);
    }

    private void cleanup() {
        getSessionProvider().cleanup(getUserDatabaseSpec());
    }

    private UserDatabaseSpec getUserDatabaseSpec() {
        return UserDatabaseSpec.fromUserAndAlias(testUser, getDatabaseAlias());
    }

    protected DatabaseAlias getDatabaseAlias() {
        return getAppDatabaseRef();
    }

    protected AppDatabaseRef getAppDatabaseRef() {
        if (!databaseRef.isPresent()) {
            databaseRef = Option.of(getSessionProvider().nextDbRef());
        }

        return databaseRef.get();
    }

    public interface DataSourceProvider {
        DataSourceType type();

        DataSource dataSource();

        DataApiUserId nextUser();

        AppDatabaseRef nextDbRef();

        SimpleDataRecord nextRecord();

        default void cleanup(UserDatabaseSpec databaseSpec) {
            // do nothing by default
        }
    }

    private SetF<SimpleDataRecord> unifyRecord(ListF<SimpleDataRecord> records) {
        return yaMoneyIsUnderTest()
                ? records.map(r -> replaceRecordId(r, "X")).unique()
                : records.unique();
    }

    private static SimpleDataRecord replaceRecordId(SimpleDataRecord record, String recordId) {
        return record.withRecordId(new SimpleRecordId(record.id().collectionId(), recordId));
    }

    private boolean yaMoneyIsUnderTest() {
        return type == DataSourceType.YA_MONEY;
    }

    @Autowired
    public void setDataSourceProviders(List<DataSourceProvider> dataSourceProviders) {
        this.dataSourceProviders = Cf.x(dataSourceProviders)
                .toMapMappingToKey(DataSourceProvider::type);
    }

    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    public @interface ExceptionIfTrait {
        DataSourceTrait trait();

        Class<? extends Throwable> exception();
    }

    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    public @interface ExceptionIfNoTrait {
        DataSourceTrait trait();

        Class<? extends Throwable> exception();
    }

    @Retention(RetentionPolicy.RUNTIME)
    @Target(ElementType.METHOD)
    public @interface SkipForType {
        DataSourceType value();
    }

    private class CustomRule implements TestRule {
        @Override
        public Statement apply(Statement base, Description description) {
            return Option.<Statement>empty()
                    .orElse(
                            () -> getStatementO(ExceptionIfTrait.class,
                                    a -> dataSourceHasTrait(a.trait()),
                                    a -> new ExpectException(base, a.exception()),
                                    description
                            )
                    )
                    .orElse(
                            () -> getStatementO(ExceptionIfNoTrait.class,
                                    a -> !dataSourceHasTrait(a.trait()),
                                    a -> new ExpectException(base, a.exception()),
                                    description
                            )
                    )
                    .orElse(
                            () -> getStatementO(SkipForType.class,
                                    a -> a.value() == type,
                                    a -> new SkipStatement("Skipping test for type = " + type),
                                    description
                            )
                    )
                    .getOrElse(base);
        }

        private <T extends Annotation> Option<Statement> getStatementO(
                Class<T> type,
                Predicate<T> predicate,
                Function<T, Statement> function,
                Description description)
        {
            return Option.ofNullable(description.getAnnotation(type))
                    .filter(predicate::test)
                    .map(function::apply);
        }
    }

    private class SkipStatement extends Statement {
        final AssumptionViolatedException ex;

        SkipStatement(String message) {
            this.ex = new AssumptionViolatedException(message);
        }

        @Override
        public void evaluate() throws Throwable {
            throw ex;
        }
    }
}
