package ru.yandex.chemodan.app.dataapi.api.datasource;

import java.util.function.Supplier;

import org.mockito.Mockito;
import org.mockito.stubbing.Answer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.DatabasesContextConfiguration;
import ru.yandex.chemodan.app.dataapi.core.dao.NotFoundHandler;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDaoImpl;
import ru.yandex.chemodan.app.dataapi.core.dao.support.DataApiRandomValueGenerator;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.PassportDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.yamoney.YaMoneyDatabaseRef;
import ru.yandex.chemodan.app.dataapi.core.datasources.yamoney.YaMoneyRecordFixtures;
import ru.yandex.chemodan.app.dataapi.core.datasources.yamoney.YamoneyDatabaseDataSource;
import ru.yandex.chemodan.util.test.TestAppsAndUsers;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Configuration
@Import(DatabasesContextConfiguration.class)
public class SessionProviderTestContextConfiguration {
    public static final NotFoundHandler.ProxyCreator notFoundHandlerProxyCreator = new NotFoundHandler.ProxyCreator() {
        @Override
        public <T> T getProxy(Class<T> clazz, T target) {
            if (target instanceof DataRecordsJdbcDaoImpl) {
                NotFoundHandler handler = new NotFoundHandler(target);
                T proxy = NotFoundHandler.ProxyCreator.getProxy(clazz, handler);
                Answer answer = inv -> handler.invoke(proxy, inv.getMethod(), inv.getArguments());
                return Mockito.mock(clazz, answer);
            } else {
                return NotFoundHandler.ProxyCreator.DEFAULT.getProxy(clazz, target);
            }
        }
    };

    @Bean
    DataSourceSessionImplTest.DataSourceProvider diskSessionProvider(DiskDataSource ds) {
        DataApiRandomValueGenerator randomValueGenerator = new DataApiRandomValueGenerator();
        return new RandomRecordDsProvider(
                DataSourceType.DISK,
                () -> new AppDatabaseRef("app", "dbId"),
                ds,
                randomValueGenerator::createDataApiUserId
        );
    }

    @Bean
    DataSourceSessionImplTest.DataSourceProvider yaMoneySessionProvider(YamoneyDatabaseDataSource dataSource) {
        DataApiPassportUserId user = getRandomRegisteredTestUser();
        return new AbstractDsProvider(
                DataSourceType.YA_MONEY,
                YaMoneyDatabaseRef.DATA::dbRef,
                dataSource,
                () -> user)
        {
            int index = 0;

            final ListF<SimpleDataRecord> records = YaMoneyRecordFixtures.DATA.getSimpleDataRecords();

            @Override
            public SimpleDataRecord nextRecord() {
                return records.get(index++ % records.size());
            }

            @Override
            public void cleanup(UserDatabaseSpec databaseSpec) {
                dataSource.openSession(databaseSpec)
                        .cleanUpAfterTests();
            }
        };
    }

    @Bean
    DataSourceSessionImplTest.DataSourceProvider passportSessionProvider(PassportDataSource dataSource) {
        return new RandomRecordDsProvider(
                DataSourceType.PASSPORT,
                new Supplier<AppDatabaseRef>() {
                    long dbId = System.currentTimeMillis();

                    @Override
                    public AppDatabaseRef get() {
                        return new AppDatabaseRef("app", "dbId" + dbId++);
                    }
                },
                dataSource,
                SessionProviderTestContextConfiguration::getRandomRegisteredTestUser
        );
    }

    @Bean
    @Primary
    NotFoundHandler.ProxyCreator notFoundHandlerProxyCreator() {
        return notFoundHandlerProxyCreator;
    }

    private abstract static class AbstractDsProvider implements DataSourceSessionImplTest.DataSourceProvider {
        private final DataSourceType type;
        private final Supplier<AppDatabaseRef> dbRefSupplier;
        private final DataSource dataSource;
        private final Supplier<DataApiUserId> testUserSupplier;

        AbstractDsProvider(DataSourceType type, Supplier<AppDatabaseRef> dbRefSupplier, DataSource ds,
                Supplier<DataApiUserId> testUserSupplier)
        {
            this.type = type;
            this.dbRefSupplier = dbRefSupplier;
            this.dataSource = ds;
            this.testUserSupplier = testUserSupplier;
        }

        @Override
        public DataSourceType type() {
            return type;
        }

        @Override
        public DataSource dataSource() {
            return dataSource;
        }

        @Override
        public final DataApiUserId nextUser() {
            return testUserSupplier.get();
        }

        @Override
        public AppDatabaseRef nextDbRef() {
            return dbRefSupplier.get();
        }
    }

    public static class RandomRecordDsProvider extends AbstractDsProvider {
        private long nextRecordId = System.currentTimeMillis();

        public RandomRecordDsProvider(DataSourceType type, Supplier<AppDatabaseRef> dbRefSupplier, DataSource ds,
                Supplier<DataApiUserId> testUserSupplier)
        {
            super(type, dbRefSupplier, ds, testUserSupplier);
        }

        @Override
        public SimpleDataRecord nextRecord() {
            return new SimpleDataRecord("testCol", "rec" + nextRecordId++);
        }
    }

    private static DataApiPassportUserId getRandomRegisteredTestUser() {
        return DataApiPassportUserId
                .fromToken(TestAppsAndUsers.randomTestingUser().getTokenFor(TestAppsAndUsers.testingApp));
    }
}
