package ru.yandex.chemodan.app.dataapi.api.db;

import java.util.function.Consumer;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContextSource;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseAliasType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.ref.internalpublic.PublicDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserType;
import ru.yandex.chemodan.app.dataapi.web.AccessForbiddenException;
import ru.yandex.chemodan.app.dataapi.web.pojo.DatabasePojo;
import ru.yandex.commune.a3.action.result.pojo.ActionResultPojo;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author tolmalev
 */
@ActionResultPojo
public class Database extends DefaultObject implements DatabaseRefSource {
    public final DataApiUserId uid;

    public final long rev;

    public final DatabaseMeta meta;

    public final DatabaseHandle dbHandle;

    public final DatabaseAccessType accessType;

    public final DatabaseAlias alias;

    public final boolean isNew;

    public final transient DatabaseTransientSettings settings;

    @BenderFlatten
    transient final ActionResult actionResult;

    public Database(Database database) {
        this(database.uid, database.dbHandle, database.rev, database.meta);
    }

    public Database(DataApiUserId uid, DatabaseHandle dbHandle, long rev, DatabaseMeta meta) {
        this(uid, dbHandle, rev, meta, DatabaseAccessType.READ_WRITE);
    }

    public Database(DataApiUserId uid, DatabaseHandle dbHandle, long rev, DatabaseMeta meta,
            DatabaseAccessType accessType)
    {
        this(uid, dbHandle, rev, meta, accessType, false);
    }

    private Database(DataApiUserId uid, DatabaseHandle dbHandle, long rev, DatabaseMeta meta,
            DatabaseAccessType accessType, boolean isNew)
    {
        this(uid, dbHandle, rev, meta, accessType, getAlias(uid, dbHandle.dbRef()), isNew,
                DatabaseTransientSettings.DEFAULT);
    }

    private static DatabaseAlias getAlias(DataApiUserId uid, DatabaseRef databaseRef) {
        return uid.type == DataApiUserType.PUBLIC
                ? new PublicDatabaseAlias((AppDatabaseRef) databaseRef)
                : databaseRef;
    }

    private Database(DataApiUserId uid, DatabaseHandle dbHandle, long rev, DatabaseMeta meta,
            DatabaseAccessType accessType, DatabaseAlias alias, boolean isNew, DatabaseTransientSettings settings)
    {
        this.uid = uid;
        this.dbHandle = dbHandle;
        this.rev = rev;
        this.meta = meta;
        this.accessType = accessType;
        this.alias = alias;
        this.actionResult = new ActionResult();
        this.isNew = isNew;
        this.settings = settings;
    }

    public static Database consNew(DataApiUserId uid, DatabaseHandle handle) {
        return consNew(uid, handle, Instant.now());
    }

    public static Database consNew(DataApiUserId uid, DatabaseHandle handle, Instant time) {
        return consNew(uid, handle, time, Option.empty());
    }

    public static Database consNew(DataApiUserId uid, DatabaseHandle handle, Instant time, Option<String> description) {
        return consNew(uid, handle, 0L, DatabaseMeta.consNew(time, description));
    }

    public static Database consNew(DataApiUserId uid, DatabaseHandle handle, long rev, DatabaseMeta meta) {
        return new Database(uid, handle, rev, meta, DatabaseAccessType.READ_WRITE, true);
    }

    public Database asExisting() {
        return change(b -> b.isNew = false);
    }

    public Database withRev(long rev) {
        return change(b -> b.rev = rev);
    }

    public Database withIncRev() {
        return withIncRev(1);
    }

    public Database withIncRev(int diff) {
        return change(b -> {
            b.rev = rev + diff;
            b.meta = meta.withModificationTime(Instant.now());
        });
    }

    public Database withAlias(DatabaseAlias alias) {
        return change(b -> {
            b.dbHandle = dbHandle.withDbRef(alias.dbRef());
            b.alias = alias;
        });
    }

    public Database withHandle(DatabaseHandle handle) {
        return change(b -> b.dbHandle = handle);
    }

    public Database withAccess(DatabaseAccessType accessType) {
        return change(b -> b.accessType = accessType);
    }

    public Database readOnly() {
        return withAccess(DatabaseAccessType.READ_ONLY);
    }

    @SuppressWarnings("unused")
    public Database readWrite() {
        return withAccess(DatabaseAccessType.READ_WRITE);
    }

    public Database withSize(DataSize dataSize) {
        return withMeta(meta.withSize(dataSize));
    }

    public Database withSizeInc(long sizeDiff) {
        return withMeta(meta.withSizeInc(sizeDiff));
    }

    public Database withIncRecordsCount(int recordsCountDiff) {
        return withMeta(meta.withIncRecordsCount(recordsCountDiff));
    }

    public Database withDescription(Option<String> newDescription) {
        return withMeta(meta.withDescription(newDescription));
    }

    public Database withModificationTime(Instant instant) {
        return withMeta(meta.withModificationTime(instant));
    }

    public Database touch() {
        return withMeta(meta.withModificationTime(Instant.now()));
    }

    private Database withMeta(DatabaseMeta meta) {
        return change(b -> b.meta = meta);
    }

    public void checkCanWrite() {
        if (accessType != DatabaseAccessType.READ_WRITE) {
            throw new AccessForbiddenException("Wrong access type: " + accessType + " != READ_WRITE");
        }
    }

    public static Function<Database, Option<String>> getAppOF() {
        return DatabaseContextSource::appNameO;
    }

    @Override
    public DatabaseRef dbRef() {
        return dbHandle.dbRef;
    }

    public DatabaseHandle getDbHandle() {
        return dbHandle;
    }

    public String aliasId() {
        return alias.aliasId();
    }

    public DatabasePojo toPojo() {
        return new DatabasePojo(this);
    }

    public UserDatabaseSpec spec() {
        return UserDatabaseSpec.fromDatabase(this);
    }

    public Database withRecordsCount(int recordsCount) {
        return withMeta(meta.withRecordsCount(recordsCount));
    }

    public Database withNowaitLock(boolean value) {
        return change(b -> b.settings = settings.withNowaitLock(value));
    }

    public String handleValue() {
        return dbHandle.handleValue();
    }

    private Database change(Consumer<Builder> changeF) {
        Builder builder = new Builder();
        changeF.accept(builder);
        return builder.build();
    }

    private class Builder {
        DataApiUserId uid = Database.this.uid;

        long rev = Database.this.rev;

        DatabaseMeta meta = Database.this.meta;

        DatabaseHandle dbHandle = Database.this.dbHandle;

        DatabaseAccessType accessType = Database.this.accessType;

        DatabaseAlias alias = Database.this.alias;

        boolean isNew = Database.this.isNew;

        DatabaseTransientSettings settings = Database.this.settings;

        Database build() {
            return new Database(uid, dbHandle, rev, meta, accessType, alias, isNew, settings);
        }
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.WITH_ANNOTATIONS)
    private class ActionResult {
        private final Database db = Database.this;

        @BenderPart(name = "app")
        public Option<String> appName = db.appNameO();

        @BenderPart(name = "rev")
        public long rev = db.rev;

        @BenderPart(name = "id")
        public String aliasId = db.alias.aliasId();

        @BenderFlatten
        public DatabaseMeta meta = db.meta;

        @SuppressWarnings("unused")
        @BenderFlatten
        public ExternalInfo externalInfo =
                db.alias.aliasType() == DatabaseAliasType.EXTERNAL
                        ? new ExternalInfo(
                                db.accessType, db.databaseId(), ((ExternalDatabaseAlias) db.alias).clientAppName())
                        : new ExternalInfo();
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.WITH_ANNOTATIONS)
    private static class ExternalInfo {
        @BenderPart(name = "accessType")
        public Option<DatabaseAccessType> accessType;

        @BenderPart(name = "originalDatabaseId")
        public Option<String> originalDatabaseId;

        @BenderPart(name = "clientApp")
        public Option<String> clientApp;

        ExternalInfo() {
            this(Option.empty(), Option.empty(), Option.empty());
        }

        ExternalInfo(DatabaseAccessType accessType, String originalDatabaseId, String clientApp) {
            this(Option.of(accessType), Option.of(originalDatabaseId), Option.of(clientApp));
        }

        ExternalInfo(Option<DatabaseAccessType> accessType, Option<String> originalDatabaseId,
                Option<String> clientApp) {
            this.accessType = accessType;
            this.originalDatabaseId = originalDatabaseId;
            this.clientApp = clientApp;
        }
    }
}
