package ru.yandex.chemodan.app.dataapi.api.db;

import org.joda.time.Instant;
import org.joda.time.format.DateTimeFormat;
import org.junit.Test;

import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.DataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DatabaseJsonSerializeTest {
    private static final Instant INSTANT = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ssZ")
            .parseDateTime("2014-10-27 12:00:00+03:00")
            .toInstant();
    private static final Database DATABASE = new Database(
            new DataApiPassportUserId(1),
            ProfileUtils.ADDRESSES_DB_REF.consHandle("123456789"),
            123,
            new DatabaseMeta(INSTANT, INSTANT, DatabaseMeta.EMPTY_DATABASE_SIZE, 0)
    );

    @Test
    public void testDatabase() {
        serializeAndCheck(DATABASE, "addresses", Tuple2List.tuple2List());
    }

    @Test
    public void testExternalDatabase() {
        ExternalDatabaseAlias extAlias = new ExternalDatabaseAlias("client-app",
                DATABASE.appNameO().get(),
                DATABASE.databaseId()
        );
        serializeAndCheck(
                DATABASE.readOnly().withAlias(extAlias),
                ".ext.profile@addresses",
                Tuple2List.fromPairs(
                        "accessType", "read-only",
                        "originalDatabaseId", "addresses",
                        "clientApp", "client-app"
                )
        );
    }

    private void serializeAndCheck(Database database, String aliasId, Tuple2List<String, Object> suffix) {
        Tuple2List<String, Object> values = Tuple2List.<String, Object>fromPairs(
                "app", "profile",
                "rev", database.rev,
                "id", aliasId,
                "created", INSTANT.getMillis(),
                "modified", INSTANT.getMillis(),
                "size", 1024,
                "recordsCount", 0

        ).plus(suffix);

        Assert.equals(
                String.format("{%s}", StringUtils.join(values.map(this::formatExpectedKeyValue), ",")),
                serialize(database)
        );
    }

    private String formatExpectedKeyValue(String key, Object o) {
        String value = o instanceof String
                ? String.format("\"%s\"", o)
                : o.toString();

        return String.format("\"%s\":%s", key, value);
    }

    private String serialize(Database database) {
        return new String(DataApiBenderUtils.mapper().serializeJson(database));
    }
}
