package ru.yandex.chemodan.app.dataapi.api.db;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author tolmalev
 */
@BenderBindAllFields
public class DatabaseMeta extends DefaultObject {
    public static final DataSize EMPTY_DATABASE_SIZE = DataSize.fromBytes(1024);

    @BenderPart(name = "created")
    public final Instant creationTime;
    @BenderPart(name = "modified")
    public final Instant modificationTime;

    public final DataSize size;
    public final long recordsCount;
    public final Option<String> description;

    public DatabaseMeta(Instant creationTime, Instant modificationTime, DataSize size, long recordsCount) {
        this(creationTime, modificationTime, size, recordsCount, Option.empty());
    }

    public DatabaseMeta(Instant creationTime, Instant modificationTime, DataSize size,
            long recordsCount, Option<String> description)
    {
        this.creationTime = creationTime;
        this.modificationTime = modificationTime;
        this.size = size;
        this.recordsCount = recordsCount;
        this.description = description;
    }

    static DatabaseMeta consNew(Instant time, Option<String> description) {
        return new DatabaseMeta(time, time, EMPTY_DATABASE_SIZE, 0, description);
    }

    public DatabaseMeta withModificationTime(Instant instant) {
        return new DatabaseMeta(creationTime, instant, size, recordsCount, description);
    }

    public DatabaseMeta withSize(DataSize size) {
        return new DatabaseMeta(creationTime, Instant.now(), size, recordsCount, description);
    }

    public DatabaseMeta withSizeInc(long sizeDiff) {
        return new DatabaseMeta(creationTime, Instant.now(),
                size.plus(DataSize.fromBytes(sizeDiff)), recordsCount, description);
    }

    public DatabaseMeta withIncRecordsCount(int recordsCountDiff) {
        return new DatabaseMeta(creationTime, Instant.now(),
                size, recordsCount + recordsCountDiff, description);
    }

    public DatabaseMeta withDescription(Option<String> newDescription) {
        return new DatabaseMeta(creationTime, Instant.now(), size, recordsCount, newDescription);
    }

    public DatabaseMeta withRecordsCount(int recordsCount) {
        return new DatabaseMeta(creationTime, Instant.now(), size, recordsCount, description);
    }
}
