package ru.yandex.chemodan.app.dataapi.api.db.filter;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContext;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContextSource;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandleRevisions;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandles;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandlesOptionSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefs;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefsSource;
import ru.yandex.chemodan.app.dataapi.api.db.revision.DatabaseRefRevisions;
import ru.yandex.misc.db.q.ConditionUtils;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public final class DatabasesFilter extends DefaultObject implements DatabaseContextSource, DatabaseHandlesOptionSource {
    private final State state;

    private final SqlLimits limits;

    public final DatabaseRefsSource excludeDbRefs;

    public DatabasesFilter(DatabaseContext context) {
        this(new ContextState(context));
    }

    public DatabasesFilter(DatabaseRefsSource refs) {
        this(new RefsState(refs));
    }

    public DatabasesFilter(DatabaseRefRevisions revs) {
        this(new RevisionsState(revs));
    }

    private DatabasesFilter(State state) {
        this(state, DatabaseRefs.empty(state.dbContext()), SqlLimits.all());
    }

    private DatabasesFilter(State state, DatabaseRefsSource excludeDbRefs, SqlLimits limits) {
        this.state = state;
        this.excludeDbRefs = excludeDbRefs;
        this.limits =  limits;
    }

    public DatabasesFilter withLimits(SqlLimits limits) {
        return new DatabasesFilter(state, excludeDbRefs, limits);
    }

    public DatabasesFilter excludeDbRefs(DatabaseRefsSource dbRefSrc) {
        return new DatabasesFilter(state, dbRefSrc, limits);
    }

    Type type() {
        return state.type();
    }

    @Override
    public DatabaseContext dbContext() {
        return state.dbContext();
    }

    @Override
    public Option<DatabaseHandles> dbHandlesO() {
        return dbRefsSrcO()
                .filterMap(DatabaseRefsSource::dbHandlesO);
    }

    private DatabaseRefs dbRefs() {
        return dbRefsSrcO()
                .map(DatabaseRefsSource::dbRefs)
                .getOrThrow(() -> new IllegalStateException("Database refs is not specified"));
    }

    public Option<DatabaseHandleRevisions> getRevisionsZippedWith(DatabaseHandles handles) {
        return state.dbRevsO()
                .map(revisions -> revisions.zipWith(handles));
    }

    public SqlCondition toSqlCondition() {
        return dbContext().toSqlCondition()
                .and(state.extraSqlCondition());
    }

    public Option<DatabaseRefsSource> dbRefsSrcO() {
        return state.dbRefsSrcO();
    }

    public SqlLimits getLimits() {
        return limits;
    }

    public ListF<String> databaseIds() {
        return dbRefs().databaseIds;
    }

    public DatabasesFilterOrAll toAllFilter() {
        return new DatabasesFilterOrAll(this);
    }

    enum Type {
        CONTEXT, REFS, REVS
    }

    private static abstract class State {
        abstract Type type();

        abstract DatabaseContext dbContext();

        abstract SqlCondition extraSqlCondition();

        Option<DatabaseRefsSource> dbRefsSrcO() {
            return Option.empty();
        }

        Option<DatabaseRefRevisions> dbRevsO() {
            return Option.empty();
        }
    }

    private static class ContextState extends State {
        final DatabaseContext context;

        private ContextState(DatabaseContext context) {
            this.context = context;
        }

        @Override
        Type type() {
            return Type.CONTEXT;
        }

        @Override
        DatabaseContext dbContext() {
            return context;
        }

        @Override
        SqlCondition extraSqlCondition() {
            return SqlCondition.trueCondition();
        }
    }

    private static class RefsState extends State {
        final DatabaseRefsSource dbRefsSrc;

        private RefsState(DatabaseRefsSource dbRefsSrc) {
            this.dbRefsSrc = dbRefsSrc;
        }

        @Override
        Type type() {
            return Type.REFS;
        }

        @Override
        DatabaseContext dbContext() {
            return dbRefsSrc.dbContext();
        }

        @Override
        Option<DatabaseRefsSource> dbRefsSrcO() {
            return Option.of(dbRefsSrc);
        }

        @Override
        SqlCondition extraSqlCondition() {
            return ConditionUtils.column("dbId").inSet(dbRefsSrc.dbRefs().databaseIds);
        }
    }

    private static class RevisionsState extends State {
        final DatabaseRefRevisions revisions;

        private RevisionsState(DatabaseRefRevisions revisions) {
            this.revisions = revisions;
        }

        @Override
        Type type() {
            return Type.REVS;
        }

        @Override
        DatabaseContext dbContext() {
            return revisions.dbContext();
        }

        @Override
        Option<DatabaseRefsSource> dbRefsSrcO() {
            return Option.of(revisions.dbRefs());
        }

        @Override
        Option<DatabaseRefRevisions> dbRevsO() {
            return Option.of(revisions);
        }

        @Override
        SqlCondition extraSqlCondition() {
            return SqlCondition.any(
                    revisions.revisions().map(dbRev ->
                            ConditionUtils.column("dbId").eq(dbRev.getDatabaseId())
                                    .and(ConditionUtils.column("rev").gt(dbRev.getRev()))
                    )
            );
        }
    }
}
