package ru.yandex.chemodan.app.dataapi.api.db.handle;

import ru.yandex.bolts.collection.IterableF;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContextHolder;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContextSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefs;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefsSource;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DatabaseHandles extends DatabaseContextHolder implements IterableF<DatabaseHandle>, DatabaseRefsSource {
    private final ListF<DatabaseHandle> handles;

    private final MapF<String, DatabaseHandle> byDatabaseId;

    private final MapF<String, DatabaseHandle> byHandle;

    public DatabaseHandles(DatabaseContextSource contextSrc, ListF<DatabaseHandle> handles) {
        super(contextSrc);
        if (!handles.forAll(handle -> handle.isBoundTo(contextSrc))) {
            throw new IllegalArgumentException("All handles MUST be bound to " + contextSrc);
        }
        this.handles = handles;
        this.byDatabaseId = this.handles.toMapMappingToKey(DatabaseRefSource::databaseId);
        this.byHandle = this.handles.toMapMappingToKey(databaseHandle -> databaseHandle.handle);
    }

    public static DatabaseHandles fromDatabaseIdHandleTuples(DatabaseContextSource contextSrc,
            ListF<Tuple2<String, String>> dbIdsWithHandles)
    {
        return new DatabaseHandles(contextSrc,
                dbIdsWithHandles.map(dbIdWithHandle -> new DatabaseHandle(contextSrc.dbContext(), dbIdWithHandle))
        );
    }

    @Override
    public DatabaseRefs dbRefs() {
        return new DatabaseRefs(context, getDatabaseIds());
    }

    @Override
    public Option<DatabaseHandles> dbHandlesO() {
        return Option.of(this);
    }

    @Override
    public IteratorF<DatabaseHandle> iterator() {
        return handles.iterator();
    }

    @Override
    public boolean isEmpty() {
        return handles.isEmpty();
    }

    public DatabaseHandles excludeRefs(DatabaseRefsSource refsSource) {
        if (refsSource.isEmpty()) {
            return this;
        }

        return new DatabaseHandles(context, handles.filter(refsSource.excludeF()));
    }

    @Override
    public ListF<String> getDatabaseIds() {
        return toList().map(DatabaseRefSource::databaseId);
    }

    public ListF<DatabaseHandle> toList() {
        return handles;
    }

    public Option<DatabaseHandle> byDatabaseIdO(String databaseId) {
        return byDatabaseId.getO(databaseId);
    }

    public Option<DatabaseHandle> byHandleO(String handle) {
        return byHandle.getO(handle);
    }
}
