package ru.yandex.chemodan.app.dataapi.api.db.ref;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseAccessType;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseExistsException;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseTransientSettings;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPublicUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class UserDatabaseSpec extends DefaultObject {
    private final DataApiUserId originalUid;
    private final DatabaseAlias databaseAlias;
    private final Option<Database> databaseO;
    private final Option<DatabaseHandle> dbHandleO;
    private final DatabaseTransientSettings settings;
    private final DatabaseAccessType databaseAccess;

    public UserDatabaseSpec(DataApiUserId originalUid, DatabaseRef databaseRef) {
        this(originalUid, databaseRef, Option.empty(), Option.empty());
    }

    private UserDatabaseSpec(DataApiUserId originalUid, DatabaseAlias databaseAlias, Option<Database> databaseO,
            Option<DatabaseHandle> dbHandleO)
    {
        this(originalUid, databaseAlias, databaseO, dbHandleO, DatabaseTransientSettings.DEFAULT
                , DatabaseAccessType.READ_WRITE);
    }
    private UserDatabaseSpec(DataApiUserId originalUid, DatabaseAlias databaseAlias, Option<Database> databaseO,
            Option<DatabaseHandle> dbHandleO, DatabaseTransientSettings settings, DatabaseAccessType databaseAccess)
        {
        this.originalUid = originalUid;
        this.databaseAlias = databaseAlias;
        this.databaseO = databaseO;
        this.dbHandleO = dbHandleO;
        this.settings = settings;
        this.databaseAccess = databaseAccess;
    }

    public static UserDatabaseSpec fromUserAndHandle(DataApiUserId originalUid, DatabaseHandle dbHandle) {
        return new UserDatabaseSpec(originalUid, dbHandle.dbRef(), Option.empty(), Option.of(dbHandle));
    }

    public static UserDatabaseSpec fromDatabase(Database db) {
        return new UserDatabaseSpec(db.uid, db.alias, Option.of(db), Option.of(db.dbHandle), db.settings, db.accessType);
    }

    public static UserDatabaseSpec fromUserAndAlias(DataApiUserId originalUid, DatabaseAlias databaseAlias) {
        DatabaseAccessType accessType =
                databaseAlias.aliasType() == DatabaseAliasType.PUBLIC && !originalUid.isPublic()
                ? DatabaseAccessType.READ_ONLY : DatabaseAccessType.READ_WRITE;
        return new UserDatabaseSpec(originalUid, databaseAlias, Option.empty(), Option.empty(),
                DatabaseTransientSettings.DEFAULT, accessType);
    }

    public DataApiUserId uid() {
        return databaseAlias.aliasType() == DatabaseAliasType.PUBLIC
                ? new DataApiPublicUserId()
                : originalUid;
    }

    public DatabaseRef databaseRef() {
        return databaseAlias.dbRef();
    }

    public DatabaseAlias databaseAlias() {
        return databaseAlias;
    }

    public Option<Database> databaseO() {
        return databaseO;
    }

    public Option<DatabaseHandle> dbHandleO() {
        return dbHandleO;
    }

    public DatabaseTransientSettings settings() {
        return settings;
    }

    public DatabaseAccessType databaseAccess() {
        return databaseAccess;
    }

    public NotFoundException consNotFound() {
        return databaseRef().consNotFound();
    }

    public NotFoundException consNotFoundDueToMissingUser() {
        return databaseRef().consNotFoundDueToMissing(uid());
    }

    public DatabaseExistsException consExistsException() {
        return databaseRef().consExistsException();
    }

    @Override
    public String toString() {
        return databaseAlias.toString();
    }

    public UserDatabaseSpec withoutDatabase() {
        return new UserDatabaseSpec(originalUid, databaseAlias, Option.empty(), dbHandleO, settings, databaseAccess);
    }
}
