package ru.yandex.chemodan.app.dataapi.api.db.ref.external;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.DataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseAccessType;
import ru.yandex.chemodan.util.ping.PingerChecker;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.primitives.registry.ZkRegistry;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author tolmalev
 */
public class ExternalDatabasesRegistry extends ZkRegistry<String, ExternalDatabasesRegistry.ExternalDatabasePojo>
        implements PingerChecker
{

    public ExternalDatabasesRegistry(ZkPath zkPath) {
        super(zkPath,
                Bender.cons(ExternalDatabasesRegistry.ExternalDatabasePojo.class, DataApiBenderUtils.getConfig()),
                ExternalDatabasesRegistry::id, s -> s
        );
    }

    private static String id(ExternalDatabasePojo pojo) {
        return id(pojo.app, pojo.originalApp, pojo.databaseId);
    }

    private static String id(ExternalDatabaseAlias external) {
        return id(external.clientAppName(), external.dbRef().appName(), external.dbRef().databaseId());
    }

    private static String id(String app, String originalAppId, String databaseId) {
        return app + ";" + originalAppId + ";" + databaseId;
    }

    public Option<DatabaseAccessType> getExternalDatabaseAccessType(ExternalDatabaseAlias external) {
        return getO(id(external)).map(pojo -> pojo.accessType);
    }

    public void removeExternalDatabase(String app, String originalAppId, String databaseId) {
        remove(id(app, originalAppId, databaseId));
    }

    public void addExternalDatabase(String app, String originalAppId, String databaseId,
            DatabaseAccessType accessType, Option<String> description)
    {
        put(new ExternalDatabasePojo(
                app, originalAppId, databaseId, accessType,
                description));
    }

    public ListF<String> getExternalIdsByInternal(String databaseId) {
        return getAll()
                .filter(extDbPojo -> extDbPojo.databaseId.equals(databaseId))
                .map(extDbPojo -> extDbPojo.toAlias().aliasId());
    }

    public ListF<ExternalDatabasePojo> list() {
        return getAll().toList();
    }

    @Override
    public boolean isActive() {
        return isInitialized();
    }

    @BenderBindAllFields
    public static final class ExternalDatabasePojo {
        public final String app;
        public final String originalApp;
        public final String databaseId;
        public final DatabaseAccessType accessType;

        public final Option<String> description;

        public ExternalDatabasePojo(String app, String originalApp, String databaseId,
                DatabaseAccessType accessType, Option<String> description)
        {
            this.app = app;
            this.originalApp = originalApp;
            this.databaseId = databaseId;
            this.accessType = accessType;
            this.description = description;
        }

        private ExternalDatabaseAlias toAlias() {
            return new ExternalDatabaseAlias(app, originalApp, databaseId);
        }
    }
}
