package ru.yandex.chemodan.app.dataapi.api.deltas;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;

/**
 * @author tolmalev
 */
public class DatabaseChange {
    public final Snapshot sourceSnapshot;
    public final Snapshot patchedSnapshot;
    public final ListF<Delta> deltas;

    public DatabaseChange(Snapshot sourceSnapshot, Snapshot patchedSnapshot) {
        this(sourceSnapshot, patchedSnapshot, Cf.list());
    }

    public DatabaseChange(Snapshot sourceSnapshot, Snapshot patchedSnapshot, ListF<Delta> deltas) {
        this.sourceSnapshot = sourceSnapshot;
        this.patchedSnapshot = patchedSnapshot;
        this.deltas = deltas;
    }

    public static DatabaseChange empty(Database database) {
        return new DatabaseChange(Snapshot.empty(database), Snapshot.empty(database), Cf.list());
    }

    public DatabaseChange filterByCollection(CollectionRef colRef) {
        RecordsFilter filter = RecordsFilter.DEFAULT.withCollectionId(colRef.collectionId());

        return new DatabaseChange(
                sourceSnapshot.filterBy(filter),
                patchedSnapshot.filterBy(filter),
                deltas.flatMap(delta -> delta.filterByCollectionO(colRef))
        );
    }

    public CollectionF<DataRecord> getSourceRecords() {
        return sourceSnapshot.records();
    }

    public CollectionF<DataRecord> getPatchedRecords() {
        return patchedSnapshot.records();
    }

    public Database sourceDatabase() {
        return sourceSnapshot.database;
    }

    public Database patchedDatabase() {
        return patchedSnapshot.database;
    }

    public ListF<Delta> getDeltas() {
        return deltas;
    }

    public SetF<DataRecordId> getDeletedIds() {
        return sourceSnapshot.recordIds()
                .minus(patchedSnapshot.recordIds());
    }

    public ListF<DataRecord> getDeletedRecords() {
        return sourceSnapshot.recordIds()
                .minus(patchedSnapshot.recordIds())
                .map(sourceSnapshot::getRecord);
    }

    public ListF<DataRecord> getDeletedRecordsWithLastRev() {
        return getDeletedRecords().map(r -> r.withNewRev(patchedSnapshot.database.rev));
    }

    public ListF<DataRecord> getNewRecords() {
        return patchedSnapshot.recordIds()
                .minus(sourceSnapshot.recordIds())
                .map(patchedSnapshot::getRecord);
    }

    public ListF<DataRecord> getUpdatedRecords() {
        return patchedSnapshot.recordIds()
                .intersect(sourceSnapshot.recordIds())
                .map(patchedSnapshot::getRecord);
    }

    public ListF<DataRecord> getNewAndUpdatedRecords() {
        return getNewRecords()
                .plus(getUpdatedRecords());
    }

    public Delta toSingleDelta() {
        return Delta.empty()
                .plusDeleted(getDeletedIds().cast())
                .plusUpdated(getUpdatedRecords())
                .plusNew(getNewRecords());
    }

    public DatabaseChange withoutRecords() {
        return new DatabaseChange(
                sourceSnapshot.withoutRecords(),
                patchedSnapshot.withoutRecords(),
                deltas);
    }

    public DatabaseChange withoutDeltas() {
        return withDeltas(Cf.list());
    }

    public DatabaseChange withDeltas(ListF<Delta> deltas) {
        return new DatabaseChange(sourceSnapshot, patchedSnapshot, deltas);
    }
}
