package ru.yandex.chemodan.app.dataapi.api.deltas;

import java.util.UUID;
import java.util.function.Function;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataApiRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.PatchableSnapshot;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.commune.protobuf5.annotation.ProtoIgnoreField;
import ru.yandex.commune.protobuf5.annotation.ProtoMessage;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author tolmalev
 */
@BenderBindAllFields
@ProtoMessage
public class Delta extends DefaultObject {
    @ProtoField(n = 1)
    public final ListF<RecordChange> changes;
    @ProtoField(n = 2)
    public final Option<String> id;
    @ProtoField(n = 3)
    public final Option<Long> rev;
    @ProtoField(n = 4)
    public final Option<Long> newRev;

    @ProtoIgnoreField
    @BenderIgnore
    public final transient boolean extra;

    public Delta(ListF<RecordChange> changes, Option<String> id, Option<Long> rev, Option<Long> newRev) {
        this(changes, id, rev, newRev, false);
    }

    private Delta(ListF<RecordChange> changes, Option<String> id, Option<Long> rev, Option<Long> newRev,
            boolean extra)
    {
        this.changes = changes;
        this.id = id;
        this.rev = rev;
        this.newRev = newRev;
        this.extra = extra;
    }

    public Delta(RecordChange change) {
        this(Cf.list(change));
    }

    public Delta(ListF<RecordChange> changes) {
        this(changes, Option.empty(), Option.empty(), Option.empty(), false);
    }

    private Delta() {
        this(Cf.list());
    }

    public static Delta empty() {
        return new Delta();
    }

    public static Delta delete(Snapshot snapshot) {
        return new Delta()
                .plusDeleted(snapshot.recordIds().cast());
    }

    public static Delta fromNewRecords(DataApiRecord... records) {
        return fromNewRecords(Cf.list(records));
    }

    public static Delta fromNewRecords(CollectionF<? extends DataApiRecord> records) {
        return empty().plusNew(records);
    }

    public Delta addTargetRevIfMissing() {
        return !newRev.isPresent()
                ? this.withNewRev(rev.get() + 1)
                : this;
    }

    public Delta withId(String id) {
        return new Delta(changes, Option.of(id), rev, newRev, extra);
    }

    public Delta withRandomId() {
        return withId(UUID.randomUUID().toString());
    }

    public Delta withRev(long rev) {
        return new Delta(changes, id, Option.of(rev), newRev, extra);
    }

    Delta withNewRev(long newRev) {
        return new Delta(changes, id, rev, Option.of(newRev), extra);
    }

    public Delta withSequentialRevs(long rev) {
        return withRev(rev)
                .withNewRev(rev + 1);
    }

    public Delta withCollectionChangesOnly(CollectionF<String> collectionIds) {
        return new Delta(changes.filter(change -> collectionIds.containsTs(change.collectionId)), id, rev, newRev,
                extra);
    }

    public Delta withoutData() {
        return new Delta(Cf.list(), id, rev, newRev, extra);
    }

    public Delta withoutDataAndWithRandomId() {
        return withoutData().withRandomId();
    }

    public Delta asExtra() {
        return new Delta(changes, id , rev, newRev, true);
    }

    public Option<Delta> filterByCollectionO(CollectionRef colRef) {
        ListF<RecordChange> changes = this.changes.filter(rc -> rc.collectionId.equals(colRef.collectionId));
        return changes.isNotEmpty()
                ? Option.of(new Delta(changes))
                : Option.empty();
    }

    public Delta plus(ListF<RecordChange> additionalChanges) {
        return new Delta(changes.plus(additionalChanges), id, rev, newRev, extra);
    }

    public Delta plusDeleted(RecordId... deletedRecordIds) {
        return plusDeleted(Cf.list(deletedRecordIds));
    }

    public Delta plusDeleted(CollectionF<RecordId> deletedRecordIds) {
        return plus(deletedRecordIds.unique().map(RecordChange::delete));
    }

    public Delta plusUpdated(ListF<? extends DataApiRecord> updatedRecords) {
        return plus(toRecordChanges(updatedRecords, DataApiRecord::toUpdateChange));
    }

    public Delta plusNew(CollectionF<? extends DataApiRecord> newRecords) {
        return plus(toRecordChanges(newRecords, DataApiRecord::toInsertChange));
    }

    private static ListF<RecordChange> toRecordChanges(CollectionF<? extends DataApiRecord> records,
            Function<DataApiRecord, RecordChange> toRecordChangeF)
    {
        return records.groupBy(record -> record.id().collectionId())
                .mapValuesWithKey((collectionId, groupRecords) ->
                        groupRecords.map(toRecordChangeF::apply))
                .values()
                .flatten();
    }

    public String getIdUnsafe() {
        return id.get();
    }

    public void apply(PatchableSnapshot snapshot) {
        changes.forEach(change -> change.applyTo(snapshot));
    }

    public static ListF<Delta> addTargetRevIfMissing(ListF<Delta> deltas) {
        return deltas.map(Delta::addTargetRevIfMissing);
    }

    public ListF<RecordId> getChangesRecordIds() {
        return changes.map(RecordChange::getSimpleRecordId);
    }

    public Long getRevUnsafe() {
        return rev.get();
    }

    public boolean updatesAnyField(SetF<String> fields) {
        return changes.exists(c -> c.updatesAnyField(fields));
    }
}
