package ru.yandex.chemodan.app.dataapi.api.deltas;

import java.sql.ResultSet;
import java.sql.SQLException;

import org.joda.time.Instant;
import org.springframework.jdbc.core.PreparedStatementCallback;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.protobuf.ProtobufDataUtils;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.JdbcDaoUtils;
import ru.yandex.chemodan.app.dataapi.core.dao.ShardPartitionDataSource;
import ru.yandex.chemodan.app.dataapi.core.dao.support.DataApiShardPartitionDaoSupport;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.q.ConditionUtils;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.db.resultSet.RowMapperSupport;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.Instrument;
import ru.yandex.misc.monica.core.blocks.Statistic;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author tolmalev
 */
public class DeltasJdbcDaoImpl extends DataApiShardPartitionDaoSupport implements MonicaContainer, DeltasJdbcDao {

    private static final int DML_BATCH_SIZE = 100;

    @MonicaMetric
    @GroupByDefault
    private final Instrument serialization = new Instrument();
    @MonicaMetric
    @GroupByDefault
    private final Instrument deserialization = new Instrument();
    @MonicaMetric
    @GroupByDefault
    private final Statistic serializedSize = new Statistic();
    @MonicaMetric
    @GroupByDefault
    private final Statistic savedRevision = new Statistic();

    public DeltasJdbcDaoImpl(ShardPartitionDataSource dataSource) {
        super(dataSource);
    }

    @Override
    @RunWithRandomTest
    public ListF<Delta> findAfterRevision(DataApiUserId uid, DatabaseHandle handle, long rev, int limit) {
        return getReadJdbcTemplate(uid).query("SELECT content from deltas_% WHERE "
                + "handle = ? AND rev >= ? ORDER BY rev LIMIT ?",
                new DeltaMapper(),
                handle.handleValue(), rev, limit);
    }

    @Override
    @RunWithRandomTest
    public Option<Long> findRevisionAfterTime(DataApiUserId uid, String handle, Instant instant, long minRevision) {
        SqlCondition condition = ConditionUtils.column("handle").eq(handle)
                .and(ConditionUtils.column("rev").gt(minRevision))
                .and(ConditionUtils.column("time").gt(instant));
        SqlOrder order = SqlOrder.orderByColumn("rev");
        SqlLimits limits = SqlLimits.first(1);
        String query = "SELECT rev from deltas_% "
                + condition.whereSql() + " " + order.toSql() + " " + limits.toMysqlLimits();
        return getReadJdbcTemplate(uid).queryForOption(query, Long.class, condition.args());
    }

    @Override
    @RunWithRandomTest
    public Option<Delta> find(DataApiUserId uid, DatabaseHandle handle, long rev) {
        return getReadJdbcTemplate(uid).queryForOption("SELECT content from deltas_% WHERE "
                + "handle = ? AND rev = ?",
                new DeltaMapper(),
                handle.handleValue(), rev);
    }

    @Override
    @RunWithRandomTest
    public void insert(DataApiUserId uid, String datastoreHandle, Delta delta) {
        Validate.some(delta.rev, "Can't save delta without rev");

        getReadJdbcTemplate(uid).updateRow("INSERT into deltas_% "
                + "(handle, rev, delta_id, content) "
                + "VALUES (?, ?, ?, ?)",
                datastoreHandle, delta.rev.get(), delta.id.getOrNull(),
                serialize(delta));

        savedRevision.update(delta.rev.get());
    }

    @Override
    public void insertBatch(DataApiUserId uid, Tuple2List<String, Delta> handlesDeltas) {
        JdbcDaoUtils.updateRowOrBatch(getJdbcTemplate(uid),
                "INSERT into deltas_% (handle, rev, delta_id, content) VALUES (?, ?, ?, ?)",
                handlesDeltas, (handleDelta) -> Cf.list(
                        handleDelta.get1(),
                        handleDelta.get2().rev.get(),
                        handleDelta.get2().id.getOrNull(),
                        serialize(handleDelta.get2())));
    }

    @Override
    @RunWithRandomTest
    public DataSize findDeltasSize(DataApiUserId uid, String datastoreHandle, long fromRev) {
        return getReadJdbcTemplate(uid).queryForObject("SELECT SUM(LENGTH(content)) as size "
                + "FROM deltas_% WHERE handle = ? AND rev >= ?",
                (rs, rowNum) -> DataSize.fromBytes(rs.getLong("size")),
                datastoreHandle, fromRev);
    }

    @Override
    @RunWithRandomTest
    public DataSize findDeltasSize(DataApiUserId uid, String datastoreHandle) {
        return getReadJdbcTemplate(uid).queryForObject("SELECT SUM(LENGTH(content)) as size "
                + "FROM deltas_% WHERE handle = ?",
                (rs, rowNum) -> DataSize.fromBytes(rs.getLong("size")),
                datastoreHandle);
    }

    @Override
    public void delete(DataApiUserId uid, String datastoreHandle, long rev) {
        getJdbcTemplate(uid).updateRow("DELETE FROM deltas_% WHERE "
                + "handle = ? AND rev = ?",
                datastoreHandle, rev);
    }

    @Override
    public void deleteBatch(DataApiUserId uid, String datastoreHandle, ListF<Long> revs) {
        getJdbcTemplate(uid).execute("DELETE FROM deltas_% WHERE "
                + "handle = ? AND rev = ?",
                (PreparedStatementCallback<Object>) ps -> {
                    for (ListF<Long> batch: revs.paginate(DML_BATCH_SIZE)) {
                        for (long rev : batch) {
                            ps.setString(1, datastoreHandle);
                            ps.setLong(2, rev);
                            ps.addBatch();
                        }
                        ps.executeBatch();
                    }
                    return null;
                });
    }

    @Override
    public void deleteBeforeRevision(DataApiUserId uid, String datastoreHandle, long rev) {
        getJdbcTemplate(uid).update("DELETE FROM deltas_% WHERE "
                + "handle = ? AND rev <= ?",
                datastoreHandle, rev);
    }
    @Override
    public void deleteBeforeRevision(DataApiUserId uid, String handle, long rev, ChunkRateLimiter limiter) {
        SqlCondition condition = ConditionUtils.column("handle").eq(handle).and(ConditionUtils.column("rev").lt(rev));
        SqlOrder order = SqlOrder.orderByColumn("rev");
        deleteAllByChunks(uid, "deltas_%", Cf.list("handle", "rev"), condition, order, limiter);
    }

    @Override
    public void deleteAllForDatabases(DataApiUserId uid, ListF<String> handles) {
        SqlCondition condition = ConditionUtils.column("handle").inSet(handles);
        getJdbcTemplate(uid).update("DELETE from deltas_%" + condition.whereSql(), condition.args());
    }

    @Override
    public void deleteAllForDatabases(DataApiUserId uid, ListF<String> handles, ChunkRateLimiter limiter) {
        SqlCondition condition = ConditionUtils.column("handle").inSet(handles);
        deleteAllByChunks(uid, "deltas_%", Cf.list("handle", "rev"), condition, limiter);
    }

    @Override
    public MetricGroupName groupName(String s) {
        return new MetricGroupName(
                "dataapi",
                new MetricName("dataapi", "dao", "delta"),
                "Deltas jdbc dao"
        );
    }

    private class DeltaMapper extends RowMapperSupport<Delta> {
        @Override
        public Delta mapRow(ResultSet rs, int rowNum) throws SQLException {
            return deserialize(rs.getBytes("content"));
        }
    }

    private byte[] serialize(Delta delta) {
        byte[] data = serialization.measure(ProtobufDataUtils.serializeDeltaF().bind(delta));
        serializedSize.update(data.length);
        return data;
    }

    private Delta deserialize(byte[] serialized) {
        return deserialization.measure(ProtobufDataUtils.parseDeltaF().bind(serialized));
    }
}
