package ru.yandex.chemodan.app.dataapi.api.deltas;

import org.joda.time.Instant;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.DataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldMarhsallerUnmarshallerTest;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.chemodan.app.dataapi.api.data.protobuf.ProtobufDataUtils;
import ru.yandex.chemodan.app.dataapi.core.dao.support.DataApiRandomValueGenerator;
import ru.yandex.commune.protobuf5.Protobuf5Serializer;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
public class DeltasMarshallingTest {
    private static final Logger logger = LoggerFactory.getLogger(DeltasMarshallingTest.class);

    BenderMapper mapper = DataApiBenderUtils.mapper();

    @Before
    public void before() {
        System.out.println("--------------Start test--------------");
        BenderParserSerializer<Delta> deltaParserSerializer = mapper.createParserSerializer(Delta.class);
        mapper.createSerializer(Delta.class);
        Protobuf5Serializer<Delta> deltaSerializer = ProtobufDataUtils.deltaSerializer;
    }

    @After
    public void after() {
        System.out.println("--------------End test--------------");
    }

    @Test
    public void insert() {
        Delta delta = new Delta(Cf.list(
                RecordChange.insert("c1", "r1", DataFieldMarhsallerUnmarshallerTest.getTestData()),
                RecordChange.insert("c2", "r1", DataFieldMarhsallerUnmarshallerTest.getTestData()),
                RecordChange.insert("c3", "r1", DataFieldMarhsallerUnmarshallerTest.getTestData()),
                RecordChange.insert("c4", "r1", DataFieldMarhsallerUnmarshallerTest.getTestData())
        ));
        testParseSerialize(delta);
    }

    @Test
    public void delete() {
        Delta delta = new Delta(Cf.list(
                RecordChange.delete("c1", "r1"),
                RecordChange.delete("c2", "r1"),
                RecordChange.delete("c1", "r3")
        ));
        testParseSerialize(delta);
    }

    @Test
    public void jsonFromFile() {
        Delta parsed = mapper.parseJson(Delta.class,
                new ClassPathResourceInputStreamSource(getClass(), "delta.json"));

        Assert.equals(
                sampleDelta(),
                parsed
        );
    }

    public static Delta sampleDelta() {
        return new Delta(Cf.list(
                RecordChange.insert("col1", "record1", Cf
                        .map("field1", DataField.decimal(123))
                        .plus1("field2", DataField.bool(true))
                        .plus1("field3", DataField.bool(false))
                        .plus1("field4", DataField.string("string"))
                        .plus1("field5", DataField.nul())
                        .plus1("field6", DataField.nan())
                        .plus1("field7", DataField.infinity())
                        .plus1("field8", DataField.negativeInfinity())
                        .plus1("field9", DataField.timestamp(new Instant(123456789)))
                        .plus1("field10", DataField.integer(125))
                        .plus1("field11", DataField.list(
                                DataField.decimal(123),
                                DataField.nul(),
                                DataField.bool(true),
                                DataField.bool(false),
                                DataField.integer(125)
                        ))
                ),
                RecordChange.update("col1", "record2", Cf.list(
                        FieldChange.delete("field1"),
                        FieldChange.put("field2", DataField.nul()),
                        FieldChange.put("field3", DataField.list(
                                DataField.decimal(123),
                                DataField.bool(true),
                                DataField.nul()
                        ))
                )),
                RecordChange.delete("col1", "record3"),
                RecordChange.set("col1", "record4", Cf
                        .map("field1", DataField.decimal(123))
                        .plus1("field2", DataField.bool(true))
                        .plus1("field3", DataField.bool(false))
                        .plus1("field4", DataField.string("string"))
                        .plus1("field5", DataField.nul())
                        .plus1("field6", DataField.nan())
                        .plus1("field7", DataField.infinity())
                        .plus1("field8", DataField.negativeInfinity())
                        .plus1("field9", DataField.timestamp(new Instant(123456789)))
                        .plus1("field10", DataField.integer(125))
                        .plus1("field11", DataField.list(
                                DataField.decimal(123),
                                DataField.nul(),
                                DataField.bool(true),
                                DataField.bool(false),
                                DataField.integer(125)
                        ))
                )
        ), Option.of("test-unique-id"), Option.empty(), Option.empty());
    }


    @Test
    public void update() {
        ListF<FieldChange> list = Cf.list(
                FieldChange.delete("f3"),
                FieldChange.put("f1", DataField.bool(true)),
                FieldChange.put("f2", DataField.bool(false))
        );
        DataApiRandomValueGenerator R = new DataApiRandomValueGenerator();
        for (DataFieldType type : DataFieldType.values()) {
            list = list.plus(FieldChange.put(Random2.R.nextString(10), R.createDataField(type)));
        }

        Delta delta = new Delta(Cf.list(
                RecordChange.update("c1", "r1", list)
        ));

        testParseSerialize(delta);
    }

    @Test
    public void hugeDelta() {
        final DataApiRandomValueGenerator R = new DataApiRandomValueGenerator();

        Delta delta = new Delta(Cf.range(0, 1000).map(integer -> R.createRecordChange()));

        testParseSerialize(delta);
    }

    private void testParseSerialize(Delta delta) {
        int times = 2;

        Instant start = Instant.now();
        byte[] bytes = null;

        for (int i = 0; i < times; i++) {
            bytes = mapper.serializeJson(delta);
            logger.info("Serialized json {} in {}", i + 1, Instant.now().getMillis() - start.getMillis());
        }
        logger.info("Serialized json in {}", Instant.now().getMillis() - start.getMillis());
        start = Instant.now();
        Delta parsed = null;
        for (int i = 0; i < times; i++) {
            parsed = mapper.parseJson(Delta.class, bytes);
        }

        logger.info("Parsed json in {}", Instant.now().getMillis() - start.getMillis());

        byte[] jsonDelta = mapper.serializeJson(delta);
        Delta parsedDelta = mapper.parseJson(Delta.class, jsonDelta);
        Assert.equals(delta, parsedDelta);

        start = Instant.now();
        for (int i = 0; i < times; i++) {
            bytes = ProtobufDataUtils.serialize(delta);
            logger.info("Serialized protobuf {} in {}", i + 1, Instant.now().getMillis() - start.getMillis());
        }
        logger.info("Serialized protobuf in {}", Instant.now().getMillis() - start.getMillis());

        start = Instant.now();

        for (int i = 0; i < times; i++) {
            parsed = ProtobufDataUtils.parseDelta(bytes);
        }
        logger.info("Parsed protobuf in {}", Instant.now().getMillis() - start.getMillis());

        Assert.equals(delta, parsed);
    }

    @Test
    @Ignore
    public void speed() {
        final DataApiRandomValueGenerator R = new DataApiRandomValueGenerator();

        ListF<Delta> deltas = Cf.range(0, 1000).map(integer ->
                new Delta(Cf.range(0, 100).map(integer1 -> R.createRecordChange())));

        Instant start = Instant.now();
        for (Delta delta : deltas) {
            mapper.serializeJson(delta);
        }
        logger.info("Serialized json in {}", Instant.now().getMillis() - start.getMillis());

        start = Instant.now();
        for (Delta delta : deltas) {
            ProtobufDataUtils.serialize(delta);
        }
        logger.info("Serialized proto in {}", Instant.now().getMillis() - start.getMillis());
    }

}
