package ru.yandex.chemodan.app.dataapi.api.deltas;

import java.util.List;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.DataNamesUtils;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.protobuf.ProtobufDataFieldValue;
import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.commune.protobuf5.annotation.ProtoMessage;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.lang.impl.DefaultObjectReflectionCache;
import ru.yandex.misc.lang.tsb.YandexToStringBuilder;

/**
 * @author tolmalev
 */
@ProtoMessage
public class FieldChange {
    @ProtoField(n = 1)
    public final FieldChangeType type;
    @ProtoField(n = 2)
    public final String key;
    @ProtoField(n = 3)
    private final Option<ProtobufDataFieldValue> value;
    @ProtoField(n = 4)
    public final Option<Integer> listIndex;
    @ProtoField(n = 5)
    public final Option<Integer> listMoveDestIndex;

    protected FieldChange(String key) {
        this.key = key;
        this.type = FieldChangeType.DELETE;
        this.value = Option.empty();
        this.listIndex = Option.empty();
        this.listMoveDestIndex = Option.empty();
    }

    protected FieldChange(String key, ProtobufDataFieldValue value) {
        this.key = key;
        this.type = FieldChangeType.PUT;
        this.value = Option.of(value);
        this.listIndex = Option.empty();
        this.listMoveDestIndex = Option.empty();
    }

    protected FieldChange(String key, int listIndex) {
        this.key = key;
        this.type = FieldChangeType.DELETE_LIST_ITEM;
        this.value = Option.empty();
        this.listIndex = Option.of(listIndex);
        this.listMoveDestIndex = Option.empty();
    }

    protected FieldChange(String key, FieldChangeType changeType, int listIndex,
            Option<Integer> listMoveDestIndex, Option<ProtobufDataFieldValue> value)
    {
        Validate.in(changeType,
                Cf.list(FieldChangeType.INSERT_LIST_ITEM, FieldChangeType.PUT_LIST_ITEM,
                        FieldChangeType.DELETE_LIST_ITEM, FieldChangeType.MOVE_LIST_ITEM));

        switch (changeType) {
            case INSERT_LIST_ITEM:
            case PUT_LIST_ITEM:
                Validate.some(value);
                Validate.none(listMoveDestIndex);
                break;
            case DELETE_LIST_ITEM:
                Validate.none(value);
                Validate.none(listMoveDestIndex);
                break;
            case MOVE_LIST_ITEM:
                Validate.none(value);
                Validate.some(listMoveDestIndex);
                break;
            default:
                Check.fail("Impossible");
        }
        this.key = key;
        this.type = changeType;
        this.value = value;
        this.listIndex = Option.of(listIndex);
        this.listMoveDestIndex = listMoveDestIndex;
    }

    public static FieldChange delete(String key) {
        return new FieldChange(key);
    }

    public static FieldChange put(String key, DataField value) {
        return new FieldChange(key, new ProtobufDataFieldValue(value.fieldType, value.value));
    }

    public static FieldChange deleteListItem(String key, int index) {
        return new FieldChange(key, index);
    }

    public static FieldChange putListItem(String key, int index, DataField value) {
        return new FieldChange(key, FieldChangeType.PUT_LIST_ITEM, index, Option.empty(),
                Option.of(ProtobufDataFieldValue.cons(value)));
    }

    public static FieldChange insertListItem(String key, int index, DataField value) {
        return new FieldChange(key, FieldChangeType.INSERT_LIST_ITEM, index, Option.empty(),
                Option.of(ProtobufDataFieldValue.cons(value)));
    }

    public static FieldChange moveListItem(String key, int index, int destIndex) {
        return new FieldChange(key, FieldChangeType.MOVE_LIST_ITEM, index, Option.of(destIndex),
                Option.empty());
    }

    public static ListF<FieldChange> putListSorted(DataFields fields) {
        return fields.data.entries()
                .sortedBy1()
                .map(FieldChange::put);
    }

    @Override
    public int hashCode() {
        return DefaultObjectReflectionCache.get(this.getClass()).hashCode(this);
    }

    @Override
    public boolean equals(Object obj) {
        return DefaultObjectReflectionCache.get(this.getClass()).equals(this, obj);
    }

    @Override
    public String toString() {
        return YandexToStringBuilder.reflectionToStringValueObject(this);
    }

    public DataField getValue() {
        DataField dataField = value.get().toDataField();
        validateValue(dataField);
        return dataField;
    }

    public void applyTo(MapF<String, DataField> data) {
        DataNamesUtils.validateId(key);

        switch(type) {
            case PUT:
                data.put(key, getValue());
                break;

            case DELETE:
                DeltaValidate.isTrue(data.containsKeyTs(key), "Can't remove non-existing field " + key);
                data.removeTs(key);
                break;

            case INSERT_LIST_ITEM:
            case PUT_LIST_ITEM:
            case MOVE_LIST_ITEM:
            case DELETE_LIST_ITEM:
                DataField value = data.getTs(key);

                DeltaValidate.isTrue(data.containsKeyTs(key), "Can't update non-existing list " + key);
                DeltaValidate.equals(value.fieldType, DataFieldType.LIST, "Can't update " + key + ". Not a list");

                @SuppressWarnings("unchecked")
                ListF<DataField> list = Cf.toArrayList((ListF<DataField>) value.value);
                switch(type) {
                    case INSERT_LIST_ITEM:
                        DeltaValidate.le(listIndex.get(), list.size());
                        list.add(listIndex.get(), getValue());
                        break;

                    case PUT_LIST_ITEM:
                        DeltaValidate.lt(listIndex.get(), list.size());
                        list.set(listIndex.get(), getValue());
                        break;

                    case MOVE_LIST_ITEM:
                        DeltaValidate.lt(listIndex.get(), list.size());
                        DeltaValidate.lt(listMoveDestIndex.get(), list.size());
                        DataField currentValue = list.remove((int) listIndex.get());
                        list.add(listMoveDestIndex.get(), currentValue);
                        break;

                    case DELETE_LIST_ITEM:
                        DeltaValidate.lt(listIndex.get(), list.size());
                        list.remove((int) listIndex.get());
                        break;

                    default:
                        Check.fail("Unexpected type error: " + type);
                }

                data.put(key, DataField.list(list));
                break;

            default:
                Check.fail("Unexpected type error: " + type);
        }
    }

    private void validateValue(DataField field) {
        Object value = field.value;
        switch (field.fieldType) {
            case INFINITY: DeltaValidate.isTrue(Double.isInfinite((Double) value)); break;
            case NEGATIVE_INFINITY: DeltaValidate.isTrue(Double.isInfinite((Double) value)); break;
            case NAN: DeltaValidate.isTrue(Double.isNaN((Double) value)); break;
            case STRING: DeltaValidate.isFalse(((String) value).contains("\u0000"), "String value contains 0x00"); break;
            case LIST:
                for (Object o : (List) value) {
                    DeltaValidate.isTrue(o instanceof DataField);
                    validateValue((DataField) o);
                }
        }
    }

    public enum FieldChangeType {
        @ProtoField(n = 1)
        DELETE,
        @ProtoField(n = 2)
        PUT,
        @ProtoField(n = 3)
        PUT_LIST_ITEM,
        @ProtoField(n = 4)
        DELETE_LIST_ITEM,
        @ProtoField(n = 5)
        INSERT_LIST_ITEM,
        @ProtoField(n = 6)
        MOVE_LIST_ITEM
    }
}
