package ru.yandex.chemodan.app.dataapi.api.deltas;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.DataNamesUtils;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataApiRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.ModifiableDataRecords;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.PatchableSnapshot;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.commune.protobuf5.annotation.ProtoIgnoreField;
import ru.yandex.commune.protobuf5.annotation.ProtoMessage;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author tolmalev
 */
@ProtoMessage
public class RecordChange extends DefaultObject {
    @ProtoField(n = 1)
    public final RecordChangeType type;
    @ProtoField(n = 2)
    public final String collectionId;
    @ProtoField(n = 3)
    public final String recordId;
    @ProtoField(n = 4)
    public final ListF<FieldChange> fieldChanges;
    @ProtoIgnoreField
    public final transient boolean forced;

    public static RecordChange insertEmpty(String collectionId, String recordId) {
        return insert(collectionId, recordId, Cf.map());
    }

    public static RecordChange insertEmpty(RecordId recordId) {
        return insertEmpty(recordId.collectionId(), recordId.recordId());
    }

    public static RecordChange insert(String collectionId, String recordId, MapF<String, DataField> data) {
        return insert(new SimpleRecordId(collectionId, recordId), data);
    }

    public static RecordChange insert(RecordId recordId, MapF<String, DataField> data) {
        return insert(recordId, new DataFields(data));
    }

    public static RecordChange insert(RecordId recordId, DataFields data) {
        return new SimpleDataRecord(recordId, data)
                .toInsertChange();
    }

    public static RecordChange set(String collectionId, String recordId, MapF<String, DataField> data) {
        return set(new SimpleRecordId(collectionId, recordId), data);
    }

    public static RecordChange set(RecordId recordId, MapF<String, DataField> data) {
        return new SimpleDataRecord(recordId, data)
                .toSetChange();
    }

    public static RecordChange setForced(RecordId recordId, MapF<String, DataField> data) {
        return new SimpleDataRecord(recordId, data)
                .toSetForcedChange();
    }

    public static RecordChange delete(RecordId recordId) {
        return delete(recordId.collectionId(), recordId.recordId());
    }

    public static RecordChange delete(CollectionRef colRef, String recordId) {
        return delete(colRef.collectionId, recordId);
    }

    public static RecordChange delete(String collectionId, String recordId) {
        return new RecordChange(RecordChangeType.DELETE, collectionId, recordId, Cf.list());
    }

    public static RecordChange update(CollectionRef colRef, String recordId, FieldChange fieldChange) {
        return update(colRef.collectionId, recordId, fieldChange);
    }

    public static RecordChange update(String collectionId, String recordId, FieldChange fieldChange) {
        return update(collectionId, recordId, Cf.list(fieldChange));
    }

    public static RecordChange update(CollectionRef colRef, String recordId, ListF<FieldChange> fieldChanges) {
        return update(colRef.collectionId, recordId, fieldChanges);
    }

    public static RecordChange update(String collectionId, String recordId, ListF<FieldChange> fieldChanges) {
        return new RecordChange(RecordChangeType.UPDATE, collectionId, recordId, fieldChanges);
    }

    public static RecordChange update(RecordId recordId, ListF<FieldChange> fieldChanges) {
        return new RecordChange(RecordChangeType.UPDATE, recordId, fieldChanges);
    }

    public RecordChange(RecordChangeType type, RecordId recordId, ListF<FieldChange> fieldChanges) {
        this(type, recordId.collectionId(), recordId.recordId(), fieldChanges);
    }

    public RecordChange(RecordChangeType type, String collectionId, String recordId, ListF<FieldChange> fieldChanges) {
        this(type, collectionId, recordId, fieldChanges, false);
    }

    private RecordChange(
            RecordChangeType type, String collectionId, String recordId,
            ListF<FieldChange> fieldChanges, boolean forced)
    {
        this.type = type;
        this.collectionId = collectionId;
        this.recordId = recordId;
        this.fieldChanges = fieldChanges;
        this.forced = forced;
    }

    public static RecordChange toPutChange(RecordChangeType changeType, DataApiRecord record) {
        return new RecordChange(changeType, record.id(), FieldChange.putListSorted(record.data()));
    }

    public static RecordChange toSetForcedChange(DataApiRecord record) {
        RecordId id = record.id();
        ListF<FieldChange> changes = FieldChange.putListSorted(record.data());

        return new RecordChange(RecordChangeType.SET, id.collectionId(), id.recordId(), changes, true);
    }

    public ListF<LocalFieldId> getFields() {
        return fieldChanges.map(fieldChange -> new LocalFieldId(collectionId, recordId, fieldChange.key));
    }

    public SimpleRecordId getRecordId() {
        return new SimpleRecordId(collectionId, recordId);
    }

    public DataRecordId getRecordId(DatabaseHandle dbHandle) {
        return new DataRecordId(dbHandle, collectionId, recordId);
    }

    public SimpleRecordId getSimpleRecordId() {
        return new SimpleRecordId(collectionId, recordId);
    }

    public Delta toDelta() {
        return new Delta(this);
    }

    public void applyTo(PatchableSnapshot snapshot) {
        new Patcher(snapshot)
                .apply();
    }

    public boolean updatesAnyField(SetF<String> fields) {
        return type == RecordChangeType.UPDATE && fieldChanges.exists(fc -> fields.containsTs(fc.key));
    }

    private class Patcher {
        final Database database;

        final ModifiableDataRecords patchedRecords;

        final DeltaValidationMode validationMode;

        final DataRecordId recordId;

        final long newRev;

        private Patcher(PatchableSnapshot snapshot) {
            this.database = snapshot.database();
            this.patchedRecords = snapshot.patchedRecords;
            this.validationMode = snapshot.validationMode;
            this.recordId = getRecordId(database.dbHandle);
            this.newRev = snapshot.nextRev();
        }

        public void apply() {
            switch (type) {
                case INSERT:
                    insert();
                    break;

                case SET:
                    set();
                    break;

                case UPDATE:
                    update();
                    break;

                case DELETE:
                    delete();
                    break;

                default:
                    Check.fail("Unknown change type " + type);
            }
        }

        private void insert() {
            DeltaValidate.isFalse(patchedRecords.contains(recordId),
                    "Can't create already existing record '" + recordId.idStr() + "'");
            applyInsertOrSet();
        }

        private void set() {
            patchedRecords.remove(recordId);
            applyInsertOrSet();
        }

        private void applyInsertOrSet() {
            DataNamesUtils.validateId(recordId);
            fieldChanges.forEach(change -> DeltaValidate.equals(FieldChange.FieldChangeType.PUT, change.type));
            applyChangesTo(Cf.hashMap());
        }

        private void update() {
            if (patchedRecords.contains(recordId)) {
                applyChangesTo(patchedRecords.get(recordId).getModifiableData());

            } else if (validationMode != DeltaValidationMode.IGNORE_NON_EXISTENT) {
                throw new DeltaUpdateOrDeleteNonExistentRecordException(
                        "Can't update non-existing record '" + recordId.idStr() + "'");
            }
        }

        private void applyChangesTo(MapF<String, DataField> data) {
            fieldChanges.forEach(change -> change.applyTo(data));
            patchedRecords.add(new DataRecord(database.uid, recordId, newRev, data));
        }

        private void delete() {
            if (patchedRecords.remove(recordId) == null
                    && validationMode != DeltaValidationMode.IGNORE_NON_EXISTENT
                    && validationMode != DeltaValidationMode.IGNORE_NON_EXISTENT_DELETION)
            {
                throw new DeltaUpdateOrDeleteNonExistentRecordException(
                        "Can't remove non-existing record '" + recordId.idStr() + "'");
            }
        }

    }
}
