package ru.yandex.chemodan.app.dataapi.api.deltas.cleaning;

import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.Duration;
import org.joda.time.Period;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.app.dataapi.DataApiYtContextConfiguration;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.DatabasesContextConfiguration;
import ru.yandex.chemodan.app.dataapi.core.dao.JdbcDaoContextConfiguration;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.utils.YtPathsUtils;
import ru.yandex.chemodan.ratelimiter.chunk.auto.AutoRateLimiterSupportContextConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.MetricsConfiguration;
import ru.yandex.chemodan.ratelimiter.chunk.auto.RateLimitersMetrics;
import ru.yandex.chemodan.util.yasm.monitor.YasmMonitor;
import ru.yandex.chemodan.util.yt.IncrementalLogMrYtRunner;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.db.shard2.ShardManager2;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.thread.InterruptedRuntimeException;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;

/**
 * @author yashunsky
 */

@Configuration
@Import({
        DataApiYtContextConfiguration.class,
        DatabasesContextConfiguration.class,
        JdbcDaoContextConfiguration.class,
        AutoRateLimiterSupportContextConfiguration.class,
})
public class DeltasCleaningContextConfiguration {
    @Value("${dataapi-worker.deltas-cleaning.yt.operation.retry.count}")
    private int ytOperationRetryCount;

    @Value("${dataapi-worker.deltas-cleaning.yt.operation.retry.delay}")
    private Duration ytOperationRetryDelay;

    @Value("${dataapi-worker.deltas-cleaning.yt.logs-root-path}")
    private String ytLogsRootPath;

    @Value("${dataapi-worker.deltas-cleaning.yt.requests-delete-period-days}")
    private int requestsDeletePeriodDays;

    @Value("${dataapi-worker.deltas-cleaning.yt.cleaning-info.inspect-period-days}")
    private int inspectPeriodDays;

    private final DataApiYtContextConfiguration ytConfig;

    @Autowired
    public DeltasCleaningContextConfiguration(DataApiYtContextConfiguration ytConfig) {
        this.ytConfig = ytConfig;
    }

    private YPath logsYtRootPath() {
        return YPath.simple(ytLogsRootPath);
    }

    @Bean
    public RetryPolicy ytDeltasCleaningOperationRetryPolicy() {
        return new RetryPolicy()
                .withMaxRetries(ytOperationRetryCount)
                .withDelay(ytOperationRetryDelay.getMillis(), TimeUnit.MILLISECONDS)
                .abortOn(Cf.list(VirtualMachineError.class, ThreadLocalTimeoutException.class,
                        InterruptedRuntimeException.class, InterruptedException.class));
    }

    @Bean
    public IncrementalLogMrYtRunner retrieveLatestRequestsRunner() {
        return new IncrementalLogMrYtRunner(
                ytConfig.dataApiYtClient(), ytDeltasCleaningOperationRetryPolicy(),
                YtPathsUtils.getDeltasCleaningYPath(),
                Period.days(requestsDeletePeriodDays), logsYtRootPath(), inspectPeriodDays,
                "retrieve_max_revisions.py", "requestedRevisions",
                Cf.list("appName", "dbId", "oauthId", "uid"),
                DeltasCleaningRoutines.class);
    }

    @Bean
    public RetrieveDbsRevisionsRunner retrieveDbsRevisionsRunner(
            Yt yt,
            DatabasesJdbcDao databasesDao,
            @Value("${dataapi-worker.deltas-cleaning.retrieve-dbs-revisions.sql-page-size}")
            int sqlPageSize,
            @Value("${dataapi-worker.deltas-cleaning.retrieve-dbs-revisions.yt-page-size}")
            int ytPageSize,
            @Value("${dataapi-worker.deltas-cleaning.yt.db-revisions-delete-period-days}")
            int deletePeriodDays,
            BazingaTaskManager bazingaTaskManager,
            DynamicDeltasCleaningControl dynamicDeltasCleaningControl)
    {
        return new RetrieveDbsRevisionsRunner(
                yt, ytDeltasCleaningOperationRetryPolicy(), YtPathsUtils.getDeltasCleaningYPath(),
                databasesDao, sqlPageSize, ytPageSize, Period.days(deletePeriodDays), bazingaTaskManager,
                dynamicDeltasCleaningControl);
    }

    @Bean
    public DynamicDeltasCleaningControl dynamicDeltasCleaningControl(
            ShardManager2 dataShardManager,
            UserMetaManager userMetaManager,
            YasmMonitor yasmMonitor,
            MetricsConfiguration metricsConfiguration,
            RateLimitersMetrics rateLimitersMetrics,
            @Value("${dataapi-worker.sharpei-host-identifier}") String sharpeiIdentifier,
            @Value("${dataapi-worker.deltas-cleaning.rate-limiter-meter-interval}") Duration meterInterval,
            @Value("${dataapi-worker.host-status-averaging-interval}") Duration averageInterval,
            @Value("${dataapi-worker.deltas-cleaning.rate-limiter-maintenance-period}") Duration maintenancePeriod,
            @Value("${dataapi.sharpei.http.max.connections}") int sharpeiMaxConnections)
    {
        return new DynamicDeltasCleaningControl(sharpeiIdentifier,
                dataShardManager, userMetaManager, yasmMonitor, metricsConfiguration, rateLimitersMetrics,
                meterInterval, averageInterval, maintenancePeriod, sharpeiMaxConnections);
    }

    @Bean
    public DeltasCleaningRoutines deltasCleaningRoutines(
            Yt yt,
            @Value("${dataapi-worker.deltas-cleaning.maps-oauth-id}")
            String mapsOauthId,
            @Value("${dataapi-worker.deltas-cleaning.yt.cleaning-info.delete-period-days}")
            int deletePeriodDays,
            @Value("${dataapi-worker.deltas-cleaning.yt.cleaning-info.inspect-period-days}")
            int inspectPeriodDays,
            @Value("${dataapi-worker.deltas-cleaning.yt.cleaning-info.keep-deltas-min-count}")
            int keepDeltasMinCount,
            @Value("${dataapi-worker.deltas-cleaning.yt.cleaning-info.keep-deltas-max-count}")
            int keepDeltasMaxCount,
            @Value("${dataapi-worker.deltas-cleaning.yt.cleaning-info.keep-deltas-duration}")
            Duration keepDeltasDuration,
            @Value("${dataapi-worker.deltas-cleaning.yt.cleaning-info.min-revision-to-clean}")
            int minRevisionToClean,
            @Value("${dataapi-worker.deltas-cleaning.yt.clean-databases-batch-size}")
            int cleanDatabasesBatchSize,
            @Value("${dataapi-worker.deltas-cleaning.delete-deltas-batch-size}")
            int deleteDeltasBatchSize,
            RetrieveDbsRevisionsRunner retrieveDbsRevisionsRunner,
            DeltasJdbcDao deltasJdbcDao,
            BazingaTaskManager bazingaTaskManager,
            DynamicDeltasCleaningControl dynamicDeltasCleaningControl,
            DeltaCleaningRegistry deltaCleaningRegistry)
    {
        return new DeltasCleaningRoutines(yt, ytDeltasCleaningOperationRetryPolicy(),
                YtPathsUtils.getDeltasCleaningYPath(),
                Period.days(deletePeriodDays), mapsOauthId, inspectPeriodDays,
                keepDeltasMinCount, keepDeltasMaxCount, keepDeltasDuration, minRevisionToClean,
                cleanDatabasesBatchSize, deleteDeltasBatchSize,
                retrieveLatestRequestsRunner(), retrieveDbsRevisionsRunner, deltasJdbcDao, bazingaTaskManager,
                dynamicDeltasCleaningControl, deltaCleaningRegistry);
    }

    @Bean
    public DeltasCleaningTask deltasCleaningTask(DeltasCleaningRoutines deltasCleaningRoutines) {
        return new DeltasCleaningTask(deltasCleaningRoutines);
    }

    @Bean
    public DeltasCleaningCronTask deltasCleaningCronTask(
            DeltasCleaningRoutines deltasCleaningRoutines,
            @Value("${dataapi-worker.deltas-cleaning.cron-task-execution-hour}")
            int executionHour)
    {
        return new DeltasCleaningCronTask(deltasCleaningRoutines, executionHour);
    }

    @Bean
    public UploadShardRevisionsToYtTask uploadRevisionsToYtTask(RetrieveDbsRevisionsRunner retrieveDbsRevisionsRunner) {
        return new UploadShardRevisionsToYtTask(retrieveDbsRevisionsRunner);
    }

    @Bean
    public DeltaCleaningRegistry deltaCleaningRegistry(@Qualifier("zkRoot") ZkPath zkRoot, ZkManager zkManager) {
        DeltaCleaningRegistry registry = new DeltaCleaningRegistry(zkRoot.child("delta-cleaning"));
        zkManager.addClient(registry);
        return registry;
    }

    @Bean
    public ScalableDeltasCleaningCronTask mainDeltasCleaningCronTask(DeltasCleaningRoutines deltasCleaningRoutines,
                                                                     DeltaCleaningRegistry deltaCleaningRegistry) {
        return new ScalableDeltasCleaningCronTask(deltasCleaningRoutines, deltaCleaningRegistry);
    }

    @Bean
    public UploadRevisionsTask uploadRevisionsTask(RetrieveDbsRevisionsRunner retrieveDbsRevisionsRunner,
                                                   DeltaCleaningRegistry deltaCleaningRegistry)
    {
        return new UploadRevisionsTask(retrieveDbsRevisionsRunner, deltaCleaningRegistry);
    }

    @Bean
    public RequestRevisionsTask requestRevisionsTask(DeltasCleaningRoutines deltasCleaningRoutines,
                                                     DeltaCleaningRegistry deltaCleaningRegistry)
    {
        return new RequestRevisionsTask(deltasCleaningRoutines, deltaCleaningRegistry);
    }
}
