package ru.yandex.chemodan.app.dataapi.api.deltas.cleaning;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.Period;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasJdbcDao;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.ratelimiter.chunk.auto.AutoUserAwareRwRateLimiter;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.chemodan.util.yt.IncrementalLogMrYtRunner;
import ru.yandex.chemodan.util.yt.TableBatchExecutor;
import ru.yandex.chemodan.util.yt.YtCleaner;
import ru.yandex.chemodan.util.yt.YtRunnerBase;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.TaskId;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.operations.Operation;
import ru.yandex.inside.yt.kosher.operations.OperationStatus;
import ru.yandex.inside.yt.kosher.operations.specs.CommandSpec;
import ru.yandex.inside.yt.kosher.operations.specs.MapReduceSpec;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadLocalTimeout;

/**
 * @author yashunsky
 */
public class DeltasCleaningRoutines extends YtRunnerBase {
    private static final Logger logger = LoggerFactory.getLogger(DeltasCleaningRoutines.class);

    private final DynamicProperty<Boolean> dryRun = new DynamicProperty<>("deltas-clening.dry-run", true);
    private final DynamicProperty<Boolean> useScalableDeltasCleaningTask = new DynamicProperty<>(
            "dataapi-delta-cleaning-scalable", false);

    private final String mapsOauth;
    private final int inspectDaysCount;
    private final int keepDeltasMinCount;
    private final int keepDeltasMaxCount;
    private final Duration keepDeltasDuration;
    private final int minDbRevision;
    private final int cleanDeltasBatchSize;
    private final int deleteBatchSize;

    private final IncrementalLogMrYtRunner requestsRunner;
    private final RetrieveDbsRevisionsRunner revisionsRunner;
    private final DeltasJdbcDao deltasJdbcDao;
    private final BazingaTaskManager bazingaTaskManager;
    private final YtCleaner ytCleaner;

    private final AutoUserAwareRwRateLimiter userAwareRateLimiter;
    private final DeltaCleaningRegistry deltaCleaningRegistry;

    public DeltasCleaningRoutines(Yt yt, RetryPolicy retryPolicy, YPath rootPath, Period deletePeriod,
                                  String mapsOauth, int inspectDaysCount, int keepDeltasMinCount, int keepDeltasMaxCount,
                                  Duration keepDeltasDuration, int minDbRevision, int cleanDeltasBatchSize,
                                  int deleteBatchSize,
                                  IncrementalLogMrYtRunner requestsRunner,
                                  RetrieveDbsRevisionsRunner revisionsRunner,
                                  DeltasJdbcDao deltasJdbcDao, BazingaTaskManager bazingaTaskManager,
                                  DynamicDeltasCleaningControl rateLimiterConfigurator,
                                  DeltaCleaningRegistry deltaCleaningRegistry)
    {
        super(yt, retryPolicy, rootPath, "join.py");
        this.ytCleaner = new YtCleaner(yt, retryPolicy, deletePeriod);
        this.mapsOauth = mapsOauth;
        this.inspectDaysCount = inspectDaysCount;
        this.keepDeltasMinCount = keepDeltasMinCount;
        this.keepDeltasMaxCount = keepDeltasMaxCount;
        this.keepDeltasDuration = keepDeltasDuration;
        this.minDbRevision = minDbRevision;
        this.cleanDeltasBatchSize = cleanDeltasBatchSize;
        this.deleteBatchSize = deleteBatchSize;
        this.requestsRunner = requestsRunner;
        this.revisionsRunner = revisionsRunner;
        this.deltasJdbcDao = deltasJdbcDao;
        this.bazingaTaskManager = bazingaTaskManager;
        this.userAwareRateLimiter = rateLimiterConfigurator.getAutoRateLimiter();
        this.deltaCleaningRegistry = deltaCleaningRegistry;
    }

    public Option<Operation> retrieveCleaningInfo(LocalDate by, boolean runRequestsRunner, boolean runRevisionsRunner) {
        ytCleaner.deleteOldAndGetLatestNotEmpty(cleaningInfoPath(), by);

        Option<Operation> requestsOperation = Option.empty();

        if (runRequestsRunner) {
            requestsOperation = requestsRunner.retrieveFromPrevCall(by);
        }

        if (runRevisionsRunner) {
            revisionsRunner.retrieve(by);
        }

        if (runRequestsRunner && !requestsRunner.tableReady(by)) {
            requestsOperation.ifPresent(Operation::awaitAndThrowIfNotSuccess);
        }

        return mergeRevisions(by);
    }

    public Option<Operation> mergeRevisions(LocalDate by) {
        ListF<YPath> inputTables = Cf.list(requestsRunner.outputPath(), revisionsRunner.dbsRevisionsPath())
                .filterMap(yt.resolveTableF(by));

        Validate.hasSize(2, inputTables, "Need both actual requests and revisions tables");

        CommandSpec mapCommandSpec = consPythonCmdWithoutUtfEncoding("map");

        CommandSpec reduceCommandSpec = consPythonCmdWithoutUtfEncoding(
                String.format("reduce %s %d %d", mapsOauth, inspectDaysCount, minDbRevision)
        );

        YPath newPath = cleaningInfoPath().child(by.toString());

        if (yt.existsWithRetries(newPath) && yt.getWithRetries(() -> yt.getRowCount(newPath)) > 0) {
            logger.info("Skipping {} as it is already not empty", newPath);
            return Option.empty();
        }

        ListF<YPath> outputTables = Cf.list(newPath);

        uploadScripts();

        MapReduceSpec mapReduceSpec = MapReduceSpec.builder()
                .setInputTables(inputTables)
                .setOutputTables(outputTables)
                .setMapperSpec(mapCommandSpec)
                .setReducerSpec(reduceCommandSpec)
                .setReduceBy(Cf.list("appName", "dbId", "uid"))
                .build();

        return Option.of(yt.getWithRetries(() -> yt.operations().mapReduceAndGetOp(
                mapReduceSpec)));
    }

    public void retrieveCleaningInfoAndScheduleCleaning(LocalDate by) {
        retrieveCleaningInfo(by, true, true).ifPresent(Operation::awaitAndThrowIfNotSuccess);
        scheduleCleaning(cleaningInfoPath().child(by.toString()), by);
    }

    public boolean scheduleRequestRevisionsTaskAndGetStatus(DeltaCleaningRegistry.DeltaCleaningPojo cleaningState) {
        LocalDate date = cleaningState.getDate();
        Instant taskStartTime = date.toDateTimeAtStartOfDay().toInstant().plus(Duration.standardHours(5));

        return scheduleTaskAndGetStatus(cleaningState.isRequestRevisionsTaskScheduled(),
                cleaningState.isRequestRevisionsTaskCompleted(),
                () -> bazingaTaskManager.schedule(new RequestRevisionsTask(date), taskStartTime),
                () -> cleaningState.withRequestRevisionsTaskScheduled(true));
    }

    public boolean scheduleUploadRevisionsTaskAndGetStatus(DeltaCleaningRegistry.DeltaCleaningPojo cleaningState) {
        LocalDate date = cleaningState.getDate();

        return scheduleTaskAndGetStatus(cleaningState.isUploadRevisionsTaskScheduled(),
                cleaningState.isUploadRevisionsTaskCompleted(),
                () -> bazingaTaskManager.schedule(new UploadRevisionsTask(date)),
                () -> cleaningState.withUploadRevisionsTaskScheduled(true));
    }

    public boolean scheduleMergeDeltaResultsAndGetStatus(DeltaCleaningRegistry.DeltaCleaningPojo cleaningState) {
        LocalDate date = cleaningState.getDate();

        return scheduleYtOperationAndGetStatus(cleaningState.getMergeDeltasOperationId(),
                () -> mergeRevisions(date),
                cleaningState::withMergeDeltasOperationId);
    }

    private boolean scheduleTaskAndGetStatus(boolean scheduled, boolean completed, Function0<FullJobId> scheduleTaskF,
                                             Function0<DeltaCleaningRegistry.DeltaCleaningPojo> updateRegistry) {
        if (completed) {
            return true;
        }

        if (scheduled) {
            return false;
        }

        scheduleTaskF.apply();
        deltaCleaningRegistry.put(updateRegistry.apply());

        return false;
    }

    private boolean scheduleYtOperationAndGetStatus(Option<String> id, Function0<Option<Operation>> runF,
                                                    Function<Option<String>, DeltaCleaningRegistry.DeltaCleaningPojo> updateRegistry)
    {
        if (id.isPresent()) {
            OperationStatus status = yt.operations()
                    .getOperation(GUID.valueOf(id.get()))
                    .getStatus();

            if (status.isSuccess()) {
                return true;
            }
            if (!status.isFinished()) {
                return false;
            }
        }

        Option<Operation> operations = runF.apply();
        if (!operations.isPresent()) {
            return true;
        }

        deltaCleaningRegistry.put(updateRegistry.apply(Option.of(operations.get().getId().toString())));
        return false;
    }

    public void scheduleAndWaitRequestRevisionYtOperation(LocalDate date, long timeoutMs) {
        while (true) {
            DeltaCleaningRegistry.DeltaCleaningPojo cleaningState = deltaCleaningRegistry.get();

            boolean completed = scheduleYtOperationAndGetStatus(cleaningState.getRequestRevisionsOperationId(),
                    () -> requestsRunner.retrieveFromPrevCall(date),
                    cleaningState::withRequestRevisionsOperationId);

            if (completed) {
                return;
            }

            try {
                Thread.sleep(timeoutMs);
            } catch (InterruptedException e) {
                throw ExceptionUtils.translate(e);
            }
        }
    }

    public void scheduleCleaning(DeltaCleaningRegistry.DeltaCleaningPojo cleaningState) {
        if (cleaningState.isCleaningScheduled()) {
            return;
        }

        LocalDate date = cleaningState.getDate();
        ytCleaner.deleteOldAndGetLatestNotEmpty(cleaningInfoPath(), date);
        scheduleCleaning(cleaningInfoPath().child(date.toString()), date);
        deltaCleaningRegistry.put(cleaningState.withCleaningScheduled(true));
    }

    public boolean isCleaningCompleted(DeltaCleaningRegistry.DeltaCleaningPojo cleaningPojo) {
        return cleaningPojo.isRequestRevisionsTaskCompleted()
                && cleaningPojo.isUploadRevisionsTaskCompleted()
                && cleaningPojo.isCleaningScheduled()
                && bazingaTaskManager.getActiveJobs(TaskId.from(DeltasCleaningTask.class), SqlLimits.first(1)).isEmpty();
    }

    public void scheduleCleaning(YPath path, LocalDate by) {
        if (yt.getRowCount(path) == 0) {
            throw new IllegalStateException("no data to schedule tasks");
        }

        TableBatchExecutor executor = new TableBatchExecutor(yt, path, cleanDeltasBatchSize);

        Instant now = by.toDateTimeAtStartOfDay().toInstant();

        executor.executeByIndexes((lowerRowIndex, upperRowIndex) ->
                bazingaTaskManager.schedule(new DeltasCleaningTask(path, lowerRowIndex, upperRowIndex, now)));
    }

    public void cleanDeltas(YPath pathWithRange, Instant now) {
        Instant keepDeltasAfterInstant = now.minus(keepDeltasDuration);

        Function1V<CleaningInfo> cleanF = info -> {
            ThreadLocalTimeout.check();
            DataApiUserId uid = DataApiUserId.parse(info.uid);

            long deleteBefore;

            if (!info.requested) {
                deleteBefore = info.rev;
            } else {
                long maxPossibleRev = info.rev - keepDeltasMinCount;
                long minPossibleRev = info.rev - keepDeltasMaxCount;


                deleteBefore = deltasJdbcDao
                        .findRevisionAfterTime(uid, info.handle, keepDeltasAfterInstant, minPossibleRev)
                        .getOrElse(maxPossibleRev);

                deleteBefore = deleteBefore > maxPossibleRev ? maxPossibleRev : deleteBefore;

                if (info.mapsRevision > -1 && info.mapsRevision < deleteBefore) {
                    deleteBefore = info.mapsRevision;
                }
            }

            logger.info("Deleting deltas for uid {} handle {} before {}", uid, info.handle, deleteBefore);
            if (!dryRun.get()) {
                try {
                    deleteBatched(uid, info.handle, info.minDelta, deleteBefore);
                } catch (Throwable t) {
                    ExceptionUtils.throwIfUnrecoverable(t);
                    logger.warn("Error while deleting deltas: {}", t);
                }
            }
        };

        yt.runWithRetries(
                () -> yt.tables().read(pathWithRange, YTableEntryTypes.bender(CleaningInfo.class), cleanF));
    }

    private void deleteBatched(DataApiUserId uid, String handle, long minDelta, long limit) {
        long deleteBefore = minDelta + deleteBatchSize;

        ChunkRateLimiter rateLimiter = userAwareRateLimiter.getWriteRateLimiter(uid);

        while (deleteBefore < limit) {
            deltasJdbcDao.deleteBeforeRevision(uid, handle, deleteBefore, rateLimiter);
            deleteBefore += deleteBatchSize;
        }

        deltasJdbcDao.deleteBeforeRevision(uid, handle, limit, rateLimiter);
    }

    @BenderBindAllFields
    private static class CleaningInfo {
        private final String handle;
        private final Option<String> appName;
        private final String dbId;
        private final String uid;
        private final long rev;
        private final long minDelta;
        private final long mapsRevision;
        private final boolean requested;

        public CleaningInfo(String handle, Option<String> appName, String dbId, String uid,
                            long rev, long minDelta, long mapsRevision, boolean requested)
        {
            this.handle = handle;
            this.appName = appName;
            this.dbId = dbId;
            this.uid = uid;
            this.rev = rev;
            this.minDelta = minDelta;
            this.mapsRevision = mapsRevision;
            this.requested = requested;
        }
    }

    public YPath cleaningInfoPath() {
        return tmpRootYPath().child("cleaningInfo");
    }

    public boolean isScalableDeltasCleaningEnabled() {
        return useScalableDeltasCleaningTask.get();
    }
}
