package ru.yandex.chemodan.app.dataapi.api.deltas.cleaning;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import net.jodah.failsafe.RetryPolicy;
import org.joda.time.LocalDate;
import org.joda.time.Period;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.chemodan.app.dataapi.core.dao.ShardPartitionLocator;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.concurrent.ExecutorUtils;
import ru.yandex.chemodan.ratelimiter.chunk.auto.HostKey;
import ru.yandex.chemodan.ratelimiter.chunk.auto.MasterSlave;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.chemodan.util.yt.IncrementalLogMrYtRunner;
import ru.yandex.chemodan.util.yt.YtCleaner;
import ru.yandex.chemodan.util.yt.YtHelper;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.OnetimeJob;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.operations.Operation;
import ru.yandex.inside.yt.kosher.tables.types.JacksonTableEntryType;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;


/**
 * @author yashunsky
 */
public class RetrieveDbsRevisionsRunner {
    private static final Logger logger = LoggerFactory.getLogger(RetrieveDbsRevisionsRunner.class);

    private final DynamicProperty<Integer> threadsCount =
            DynamicProperty.cons("datasync-deltas-cleaning-dbs-export-threads-count", 4);

    private final DynamicProperty<ListF<Integer>> ignoredShards =
            DynamicProperty.cons("datasync-deltas-cleaning-dbs-export-ignored-shards", Cf.list());

    private final DynamicProperty<Long> waitBazingaTimeoutMs =
            DynamicProperty.cons("datasync-deltas-cleaning-wait-bazinga-timeout-ms", 1000L);

    private final YPath rootPath;
    private final YtHelper yt;
    private final DatabasesJdbcDao databasesDao;
    private final int sqlPageSize;
    private final int ytPageSize;
    private final YtCleaner ytCleaner;
    private final RetryPolicy retryPolicy;
    private final BazingaTaskManager bazingaTaskManager;
    private final DynamicDeltasCleaningControl dynamicDeltasCleaningControl;
    protected static final JsonNodeFactory jsonNodeFactory = new JsonNodeFactory(false);

    public RetrieveDbsRevisionsRunner(
            Yt yt, RetryPolicy retryPolicy, YPath rootPath,
            DatabasesJdbcDao databasesDao, int sqlPageSize, int ytPageSize,
            Period deletePeriod, BazingaTaskManager bazingaTaskManager,
            DynamicDeltasCleaningControl dynamicDeltasCleaningControl) {
        this.rootPath = rootPath;
        this.retryPolicy = retryPolicy;
        this.yt = new YtHelper(yt, retryPolicy);
        this.ytCleaner = new YtCleaner(yt, retryPolicy, deletePeriod);
        this.databasesDao = databasesDao;
        this.sqlPageSize = sqlPageSize;
        this.ytPageSize = ytPageSize;
        this.bazingaTaskManager = bazingaTaskManager;
        this.dynamicDeltasCleaningControl = dynamicDeltasCleaningControl;
    }

    public void retrieve(LocalDate by) {
        ytCleaner.deleteOldAndGetLatestNotEmpty(dbsRevisionsPath(), by);
        MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_ASYNC_S, () -> uploadAllDbRevisionsToYt(by));
    }

    public void scalableRetrieve(LocalDate by) {
        ytCleaner.deleteOldAndGetLatestNotEmpty(dbsRevisionsPath(), by);
        uploadAllDbRevisionsToYtByShards(by);
    }

    private String getShardTableName(int shardId) {
        return "shard" + StringUtils.leftPad(String.valueOf(shardId), 3, '0');
    }

    private YPath getShardsResultFolder() {
        return dbsRevisionsPath().child("shards");
    }

    private YPath getShardResultPath(int shardId) {
        return getShardsResultFolder()
                .child(getShardTableName(shardId));
    }

    private YPath getShardFolder(int shardId) {
        return getShardsResultFolder()
                .child(getShardTableName(shardId) + "parts");
    }

    private YPath getShardPartPath(ShardPartitionLocator locator) {
        return getShardFolder(locator.shardId)
                .child("partition" + StringUtils.leftPad(String.valueOf(locator.getPartNo()), 3, '0'))
                .append(true);
    }

    private YPath getPartsFolder() {
        return dbsRevisionsPath().child("parts");
    }

    private YPath getPartPath(ShardPartitionLocator locator) {
        String partTableName = "shard" + StringUtils.leftPad(String.valueOf(locator.shardId), 3, '0') +
                "partition" + StringUtils.leftPad(String.valueOf(locator.getPartNo()), 3, '0');
        return getPartsFolder().child(partTableName).append(true);
    }

    private void uploadShardPartition(ShardPartitionLocator locator, YPath path) {
        if (yt.existsWithRetries(path)) {
            long rowCount = yt.getWithRetries(() -> yt.getRowCount(path));

            if ((rowCount % ytPageSize) > 0) {
                logger.info("Skipping {} as it contains {} rows", path, rowCount);
                return;
            }
        }

        yt.runWithRetries(() -> {
            yt.remove(path);
            yt.cypress().create(path, CypressNodeType.TABLE, true, true);
        });

        Function1V<ListF<DbRevisionPojo>> submitToYtF = dbs -> yt.runWithRetries(
                () -> yt.tables().write(path, new JacksonTableEntryType(), dbs.map(DbRevisionPojo::asJsonNode)));

        new PartitionDbRevisionsIterator(databasesDao, locator, sqlPageSize, retryPolicy,
                dynamicDeltasCleaningControl
                        .getAutoRateLimiter()
                        .getReadRateLimiter(new HostKey(locator.getShardId(), MasterSlave.SLAVE)))
                .paginate(ytPageSize).forEachRemaining(submitToYtF);
    }

    public void uploadShard(int shardId) {
        uploadShardPartitions(getShardResultPath(shardId), getShardFolder(shardId),
                databasesDao.getShardPartitions(shardId).zipWith(this::getShardPartPath));
    }

    public void uploadShardPartitions(YPath resultPath, YPath tpmFolder,
                                      Tuple2List<ShardPartitionLocator, YPath> partitions)
    {
        if (!yt.isTableEmpty(resultPath)) {
            logger.info("Skipping {} as it is already not empty", resultPath);
            return;
        }

        ExecutorService executorService = Executors.newFixedThreadPool(threadsCount.get());
        logger.info("Sending partition tasks to executor");

        ListF<CompletableFuture<Void>> futures = partitions.map(
                (locator, path) -> ExecutorUtils.submitWithYcridForwarding(() -> {
                    logger.info("Exporting shard {} partition {}", locator.shardId, locator.getPartNo());
                    new RetryManager<>().withRetryPolicy(retryPolicy).run(
                            () -> MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_ASYNC_SS,
                                    () -> uploadShardPartition(locator, path))
                    );

                    logger.info("Shard {} partition {} exported", locator.shardId, locator.getPartNo());
                }, executorService));

        try {
            logger.info("Waiting for partition tasks to be completed");
            CompletableFutures.allOf(futures).get();
        } catch (Exception e) {
            logger.warn("Error while waiting for partition tasks");
            executorService.shutdownNow();
            throw ExceptionUtils.translate(e);
        }

        logger.info("Parts exported");
        executorService.shutdown();

        ListF<YPath> partsPaths = partitions.get2();

        logger.info("Merging parts");
        Operation merge = yt.operations().mergeAndGetOp(partsPaths, resultPath);
        merge.await();
        if (!merge.getStatus().isSuccess()) {
            throw new RuntimeException("Failed to merge parts. Operation id: " + merge.getId());
        }

        logger.info("Removing shard folder");
        yt.runWithRetries(() -> {
            yt.remove(tpmFolder);
        });
    }

    private void uploadAllDbRevisionsToYt(LocalDate by) {
        YPath resultPath = dbsRevisionsPath()
                .child(by.toString())
                .withAdditionalAttributes(IncrementalLogMrYtRunner.COMPRESSION_ATTRIBUTES);

        Tuple2List<ShardPartitionLocator, YPath> locatorsAndPaths =
                databasesDao.getShardPartitions()
                        .filterNot(locator -> ignoredShards.get().containsTs(locator.shardId))
                        .sortedBy(ShardPartitionLocator::getPartNo).zipWith(this::getPartPath);

        uploadShardPartitions(resultPath, getPartsFolder(), locatorsAndPaths);
    }

    private void uploadAllDbRevisionsToYtByShards(LocalDate by) {
        YPath resultPath = dbsRevisionsPath()
                .child(by.toString())
                .withAdditionalAttributes(IncrementalLogMrYtRunner.COMPRESSION_ATTRIBUTES);

        if (!yt.isTableEmpty(resultPath)) {
            logger.info("Skipping {} as it is already not empty", resultPath);
            return;
        }

        ListF<Integer> shards = databasesDao.getShards().filterNot(shardId -> ignoredShards.get().containsTs(shardId));

        ListF<FullJobId> tasks = shards
                .map(shardId -> bazingaTaskManager.schedule(new UploadShardRevisionsToYtTask(shardId)));
        waitBazingaTasks(tasks, waitBazingaTimeoutMs.get());
        logger.info("All shards were uploaded");

        Operation merge = yt.operations().mergeAndGetOp(shards.map(this::getShardResultPath), resultPath);
        merge.await();
        if (!merge.getStatus().isSuccess()) {
            throw new RuntimeException("Failed to merge shard results. Operation id: " + merge.getId());
        }

        logger.info("Removing shards folder");
        yt.runWithRetries(() -> {
            yt.remove(getShardsResultFolder());
        });
    }

    private void waitBazingaTasks(ListF<FullJobId> tasks, long timeoutMs) {
        // хотим побыстрее пофейлить таску и начать ретраить
        boolean hasFailedTasks = false;
        while (tasks.size() > 0) {
            logger.info("Sleep for {} millis to wait {} tasks", timeoutMs, tasks.size());

            try {
                Thread.sleep(timeoutMs);
            } catch (InterruptedException e) {
                throw ExceptionUtils.translate(e);
            }

            ListF<Option<OnetimeJob>> jobs = tasks.map(bazingaTaskManager::getOnetimeJob);
            hasFailedTasks = hasFailedTasks || jobs.stream().anyMatch(this::isOnetimeTaskFailed);

            tasks = jobs.filter(Option::isPresent)
                    .filter(job -> !this.isOnetimeTaskCompleted(job))
                    .map(job -> job.get().getId());
        }

        if (hasFailedTasks) {
            throw new RuntimeException("Some subtask failed");
        }
    }

    private boolean isOnetimeTaskCompleted(Option<OnetimeJob> task) {
        // считаем все терминальные статусы кроме EXPIRED
        return task.isEmpty() || task.get().getValue().getStatus() == JobStatus.COMPLETED
                || task.get().getValue().getStatus() == JobStatus.FAILED;
    }

    private boolean isOnetimeTaskFailed(Option<OnetimeJob> task) {
        // если во время ретрая успело удалиться из базинги, то считаем, что таска completed
        return task.isPresent() && task.get().getValue().getStatus() == JobStatus.FAILED;
    }

    public YPath dbsRevisionsPath() {
        return rootPath.child("dbsRevisions");
    }
}
