package ru.yandex.chemodan.app.dataapi.api.deltas.cleaning;

import org.joda.time.Duration;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;

import ru.yandex.chemodan.app.dataapi.worker.DataApiCronTask;
import ru.yandex.chemodan.util.TimeUtils;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.schedule.Schedule;
import ru.yandex.commune.bazinga.scheduler.schedule.ScheduleCron;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.time.MoscowTime;

/**
 * @author friendlyevil
 */
public class ScalableDeltasCleaningCronTask extends DataApiCronTask {
    private static final Logger logger = LoggerFactory.getLogger(ScalableDeltasCleaningCronTask.class);

    private final DynamicProperty<LocalTime> prepareDataForCleaningAfter =
            new DynamicProperty<>("dataapi-deltas-cleaning-start-after",
                    new LocalTime(12, 0));
    private final DynamicProperty<LocalTime> prepareDataForCleaningBefore =
            new DynamicProperty<>("dataapi-deltas-cleaning-start-before",
                    new LocalTime(23, 59, 59));

    private final DeltasCleaningRoutines deltasCleaningRoutines;
    private final DeltaCleaningRegistry deltaCleaningRegistry;

    public ScalableDeltasCleaningCronTask(DeltasCleaningRoutines deltasCleaningRoutines,
                                          DeltaCleaningRegistry deltaCleaningRegistry) {
        this.deltasCleaningRoutines = deltasCleaningRoutines;
        this.deltaCleaningRegistry = deltaCleaningRegistry;
    }

    @Override
    public Schedule cronExpression() {
        return new ScheduleCron("*/30 * * * *", MoscowTime.TZ);
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(10);
    }

    @Override
    public void execute(ExecutionContext executionContext) throws Exception {
        if (!deltasCleaningRoutines.isScalableDeltasCleaningEnabled()) {
            return;
        }

        DeltaCleaningRegistry.DeltaCleaningPojo cleaningState = deltaCleaningRegistry.get();

        if (deltasCleaningRoutines.isCleaningCompleted(cleaningState)) {
            DeltasCleaningMetrics.cleaningLag.set(0L);
            deltaCleaningRegistry.put(new DeltaCleaningRegistry.DeltaCleaningPojo());
        }

        boolean cleaningStarted = cleaningState.isStarted();
        if (!cleaningStarted && !canStartPreparingData()) {
            logger.info("We can upload cleaning data only between {} and {}",
                    prepareDataForCleaningAfter.get(), prepareDataForCleaningBefore.get());
            executionContext.setExecutionInfo(new DeltasCleaningStats());
            return;
        }

        if (!cleaningStarted) {
            cleaningState = cleaningState.withDateTime(LocalDateTime.now()).withStarted(true);
            deltaCleaningRegistry.put(cleaningState);
        }

        boolean requestRevisionsComplete = deltasCleaningRoutines.scheduleRequestRevisionsTaskAndGetStatus(cleaningState);
        boolean uploadRevisionsComplete = deltasCleaningRoutines.scheduleUploadRevisionsTaskAndGetStatus(cleaningState);

        if (requestRevisionsComplete && uploadRevisionsComplete) {
            if (deltasCleaningRoutines.scheduleMergeDeltaResultsAndGetStatus(cleaningState)) {
                deltasCleaningRoutines.scheduleCleaning(cleaningState);
            }
        }

        // update state after scheduling tasks
        cleaningState = deltaCleaningRegistry.get();
        DeltasCleaningMetrics.cleaningLag.set(getLag(cleaningState));
        executionContext.setExecutionInfo(getCleaningStats(cleaningState));
    }

    private long getLag(DeltaCleaningRegistry.DeltaCleaningPojo state) {
        if (!state.isStarted() || state.getDateTime() == null) {
            return 0;
        }

        return new Duration(state.getDateTime().toDateTime(), LocalDateTime.now().toDateTime()).getMillis();
    }

    private boolean canStartPreparingData() {
        return TimeUtils.nowBetween(prepareDataForCleaningAfter.get(), prepareDataForCleaningBefore.get());
    }

    private DeltasCleaningStats getCleaningStats(DeltaCleaningRegistry.DeltaCleaningPojo state) {
        return new DeltasCleaningStats(
                DeltasCleaningStats.CleaningStageStatus.getStatus(
                        state.isRequestRevisionsTaskScheduled(), state.isRequestRevisionsTaskCompleted()),
                DeltasCleaningStats.CleaningStageStatus.getStatus(
                        state.isUploadRevisionsTaskScheduled(), state.isUploadRevisionsTaskCompleted()),
                DeltasCleaningStats.CleaningStageStatus.getStatus(
                        state.getMergeDeltasOperationId().isPresent(), state.isCleaningScheduled()),
                DeltasCleaningStats.CleaningStageStatus.getStatus(state.isCleaningScheduled(), false)
        );
    }

}
