import sys
import json
import itertools

from time import time


REDUCE_KEYS = ("appName", "dbId", "uid")
ACCESS_KEYS = ("lastDeltaAccess", "lastDeltaRevisionAccess",
               "lastSnapshotAccess")
MAPS_ACCESS_KEYS = tuple(("%sMaps" % key for key in ACCESS_KEYS))
REQUESTS_ONLY_KEYS = tuple(itertools.chain(ACCESS_KEYS, ("oauthId",)))
REVISION_ONLY_KEYS = ("handle", "rev", "minDelta")

REQUESTS_KEYS = tuple(itertools.chain(REDUCE_KEYS, REQUESTS_ONLY_KEYS))
REVISION_KEYS = tuple(itertools.chain(REDUCE_KEYS, REVISION_ONLY_KEYS))

MAP_KEYS = tuple(itertools.chain(REDUCE_KEYS, REVISION_ONLY_KEYS,
                                 REQUESTS_ONLY_KEYS))


def do_decode(lines):
    for line in lines:
        yield json.loads(line)


def do_map(rows, *args):
    for row in rows:
        print json.dumps({key: row.get(key) for key in MAP_KEYS})


class ReduceGroup(object):
    def __init__(self, maps_oauth, interval, min_deltas_count):
        super(ReduceGroup, self).__init__()

        self.maps_oauth = maps_oauth
        self.edge = time() - (interval * 24 * 3600)
        self.min_deltas_count = min_deltas_count

        self.app_name = None
        self.db_id = None
        self.handle = None
        self.uid = None
        self.db_rev = -1
        self.db_min_delta = -1

        self.maps_revision = -1
        self.requested = False

    def add_row(self, row):
        if (row['rev'] is not None) and (row['rev'] > self.db_rev):
            self.handle = row['handle']
            self.app_name = row['appName']
            self.db_id = row['dbId']
            self.uid = row['uid']
            self.db_rev = row['rev']
            self.db_min_delta = row['minDelta']
            return

        if row['oauthId'] == self.maps_oauth:
            self.maps_revision = row['lastDeltaRevisionAccess']

        if (row['lastDeltaAccess'] > self.edge or row['lastSnapshotAccess'] > self.edge):
            self.requested = True

    def print_reduced_if_needed(self):
        deltas_count = self.db_rev - self.db_min_delta
        if (deltas_count > self.min_deltas_count and self.uid > 0 and
            None not in (self.app_name, self.db_id, self.uid)):
            print json.dumps({"appName": self.app_name,
                              "dbId": self.db_id,
                              "handle": self.handle,
                              "uid": self.uid,
                              "rev": self.db_rev,
                              "minDelta": self.db_min_delta,
                              "mapsRevision": self.maps_revision,
                              "requested": self.requested})


def do_reduce(rows, *args):
    if not args or len(args) != 3:
        print >> sys.stderr, 'Please, specify maps oauth,',
        print >> sys.stderr, 'checked interval (days)',
        print >> sys.stderr, 'and min db revision'
        sys.exit(1)

    maps_oauth = args[0]
    interval, min_rev = map(int, args[1:])

    group = ReduceGroup(maps_oauth, interval, min_rev)

    prev_id = tuple()
    for row in itertools.chain(rows, ({},)):
        row_id = tuple((row.get(key) for key in REDUCE_KEYS))
        if prev_id and prev_id != row_id:
            group.print_reduced_if_needed()
            group = ReduceGroup(maps_oauth, interval, min_rev)

        if row:
            group.add_row(row)

        prev_id = row_id


if __name__ == '__main__':
    stream = sys.stdin
    stream = do_decode(stream)
    actions = {'map': do_map, 'reduce': do_reduce}
    if len(sys.argv) < 2 or sys.argv[1] not in actions:
        print >> sys.stderr, 'Please, specify proper map or reduce',
        print >> sys.stderr, 'command to execute'
        sys.exit(1)

    actions[sys.argv[1]](stream, *sys.argv[2:])
