package ru.yandex.chemodan.app.dataapi.api.schema;

import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.GlobalDatabaseRef;
import ru.yandex.chemodan.util.exception.SchemaValidationException;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
public class SchemaValidationTest {
    private static final DatabaseSchema profileSchema = DatabaseSchema.withSingleCollection(
            new GlobalDatabaseRef("common_places")
                    .consColRefWithDbId(),

            new FieldInfo("latitude", true, DataFieldType.DECIMAL),
            new FieldInfo("longitude", true, DataFieldType.DECIMAL),
            new FieldInfo("title", true, DataFieldType.STRING),

            new FieldInfo("address_line", false, DataFieldType.STRING, new StringValueValidator(1)),
            new FieldInfo("address_line_short", false, DataFieldType.STRING, new StringValueValidator(1)),

            new FieldInfo("created", true, DataFieldType.TIMESTAMP),
            new FieldInfo("modified", true, DataFieldType.TIMESTAMP),
            new FieldInfo("last_used", true, DataFieldType.TIMESTAMP),

            new FieldInfo("tags", false, DataFieldType.LIST, new ListValueValidator(new StringValueValidator(3, 256))),

            new FieldInfo("mined_attributes", false, DataFieldType.LIST,
                    new ListValueValidator(new StringValueValidator(0))
            )
    );

    @Test
    public void validateFieldSimple() {
        new FieldInfo("", true, DataFieldType.DECIMAL)
                .validateField(Option.of(DataField.decimal(12.0)));

        new FieldInfo("", false, DataFieldType.INTEGER)
                .validateField(Option.of(DataField.integer(12)));

        new FieldInfo("", false, DataFieldType.INTEGER)
                .validateField(Option.empty());

        try {
            new FieldInfo("", true, DataFieldType.INTEGER)
                    .validateField(Option.empty());
            Assert.fail();
        } catch (SchemaValidationException e) {
        }

        try {
            new FieldInfo("", true, DataFieldType.INTEGER)
                    .validateField(Option.of(DataField.decimal(12)));
            Assert.fail();
        } catch (SchemaValidationException e) {
        }
    }

    @Test
    public void validateFieldComplex() {
        FieldInfo fieldInfo =
                new FieldInfo("", true, DataFieldType.STRING, new StringValueValidator(3, 10));

        fieldInfo.validateField(Option.of(DataField.string("123")));
        fieldInfo.validateField(Option.of(DataField.string("1234567890")));

        try {
            fieldInfo.validateField(Option.of(DataField.string("12")));
            Assert.fail();
        } catch (SchemaValidationException e) {
        }

        try {
            fieldInfo.validateField(Option.of(DataField.string("12345678901")));
            Assert.fail();
        } catch (SchemaValidationException e) {
        }
    }

    @Test
    public void validateListField() {
        FieldInfo fieldInfo = new FieldInfo("", true, DataFieldType.LIST,
                        new ListValueValidator(new StringValueValidator(3, 10)));

        fieldInfo.validateField(Option.of(DataField.list(Cf.list(
                DataField.string("123"), DataField.string("1234"), DataField.string("12345"),
                DataField.string("12345678"), DataField.string("123456790")
        ))));

        try {
            fieldInfo.validateField(Option.of(DataField.list(Cf.list(
                    DataField.string("12")
            ))));
            Assert.fail();
        } catch (SchemaValidationException e) {
        }

        try {
            fieldInfo.validateField(Option.of(DataField.list(Cf.list(
                    DataField.string("12345678901")
            ))));
            Assert.fail();
        } catch (SchemaValidationException e) {
        }
    }

    @Test
    public void validate() {
        profileSchema.validateRecord("common_places", Cf
                .map("latitude", DataField.decimal(12.0))
                .plus1("longitude", DataField.decimal(12.0))
                .plus1("title", DataField.string("test"))
                .plus1("created", DataField.timestamp(Instant.now()))
                .plus1("modified", DataField.timestamp(Instant.now()))
                .plus1("last_used", DataField.timestamp(Instant.now()))
        );

        profileSchema.validateRecord("common_places", Cf
                .map("latitude", DataField.decimal(12.0))
                .plus1("longitude", DataField.decimal(12.0))
                .plus1("title", DataField.string("test"))
                .plus1("address_line",
                        DataField.string("Россия, Санкт-Петербург, Колпино, Тверская улица, 6"))
                .plus1("address_line_short", DataField.string("Тверская улица, 6"))
                .plus1("created", DataField.timestamp(Instant.now()))
                .plus1("modified", DataField.timestamp(Instant.now()))
                .plus1("last_used", DataField.timestamp(Instant.now()))
                .plus1("tags",
                        DataField.list(Cf.list(DataField.string("home"), DataField.string("work"))))
                .plus1("mined_attributes", DataField.list(Cf.list(
                        DataField.string("address_line"),
                        DataField.string("address_line_short"))))
        );
    }
}
