package ru.yandex.chemodan.app.dataapi.apps.profile;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseAppContext;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.schema.DatabaseSchema;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.GenericApplicationManager;
import ru.yandex.chemodan.app.dataapi.apps.HandleGenerator;
import ru.yandex.chemodan.app.dataapi.apps.SpecificApplicationManager;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettingsRegistry;
import ru.yandex.chemodan.util.exception.SchemaValidationException;

/**
 * @author Denis Bakharev
 */
public class ProfileApplicationManager implements SpecificApplicationManager {
    private static final HandleGenerator handleGenerator =
            HandleGenerator.permanent(ProfileUtils::buildProfileDatabaseHandle);

    private final GenericApplicationManager genericManager;
    private final ProfileDatabaseSchemaRegistry schemaRegistry;
    private final TypeSettingsRegistry typeSettingsRegistry;

    ProfileApplicationManager(GenericApplicationManager genericManager, ProfileDatabaseSchemaRegistry schemaRegistry,
            TypeSettingsRegistry typeSettingsRegistry)
    {
        this.genericManager = genericManager;
        this.schemaRegistry = schemaRegistry;
        this.typeSettingsRegistry = typeSettingsRegistry;
    }

    @Override
    public DatabaseAppContext getApplication() {
        return ProfileUtils.PROFILE_CTX;
    }

    @Override
    public HandleGenerator getHandleGenerator() {
        return handleGenerator;
    }

    @Override
    public void validateDatabaseCreation(DataApiUserId uid, DatabaseRef databaseRef) {
        genericManager.validateDatabaseCreation(uid, databaseRef);

        validateDbRef(databaseRef);
    }

    @Override
    public void validateDatabaseUpdate(DatabaseChange change) {
        genericManager.validateDatabaseUpdate(change);

        validateProfileSchema(change);
    }

    private void validateDbRef(DatabaseRef ref) {
        if (!schemaRegistry.hasSchema(ref) && !typeSettingsRegistry.hasSettings(ref)) {
            throw new SchemaValidationException("Unknown database " + ref.databaseId());
        }
    }

    private void validateProfileSchema(DatabaseChange change) {
        DatabaseRef dbRef = change.patchedDatabase().dbRef();
        Option<DatabaseSchema> schemaO = schemaRegistry.getSchemaO(dbRef);
        change.getPatchedRecords().forEach(schemaO.isPresent()
                ? schemaO.get()::validateRecord
                : record -> settingsExists(dbRef, record)
        );
    }

    private void settingsExists(DatabaseRef dbRef, DataRecord record) {
        CollectionRef colRef = dbRef.consColRef(record.getCollectionId());
        if (!typeSettingsRegistry.hasSettings(colRef)) {
            throw new SchemaValidationException("Unknown profile collection " + record.getCollectionId());
        }
    }
}
