package ru.yandex.chemodan.app.dataapi.apps.profile;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordRef;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseExistsException;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.OutdatedChangeException;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.thread.ThreadUtils;

/**
 * @author tolmalev
 */
public abstract class ProfileManagerBase {
    private static final Logger logger = LoggerFactory.getLogger(ProfileManagerBase.class);

    private static final ListF<Integer> RETRY_DELAYS = Cf.list(10, 10, 50, 100, 500, 1000);

    private final DataApiManager dataApiManager;

    private final DiskDataSource diskDataSource;

    public ProfileManagerBase(DataApiManager dataApiManager, DiskDataSource diskDataSource) {
        this.dataApiManager = dataApiManager;
        this.diskDataSource = diskDataSource;
    }

    protected ListF<DataRecord> getRecords(DataApiUserId uid) {
        return diskDataSource.getRecords(uid, ProfileUtils.DB_REFS);
    }

    protected ListF<DataRecord> getRecords(DataApiUserId uid, DatabaseRef dbRef) {
        return dataApiManager.getRecords(new UserDatabaseSpec(uid, dbRef));
    }

    protected ListF<DataRecord> getRecords(DataApiUserId uid, CollectionRef colRef) {
        return dataApiManager.getRecords(
                new UserDatabaseSpec(uid, colRef.dbRef()),
                RecordsFilter.DEFAULT.withColRef(colRef)
        );
    }

    protected Option<DataRecord> getRecord(DataApiUserId uid, RecordRef recId) {
        return dataApiManager.getRecord(uid, recId);
    }

    protected Database getDatabase(DataApiUserId uid, DatabaseRef dbRef) {
        return getDatabase(new UserDatabaseSpec(uid, dbRef));
    }

    private Database getDatabase(UserDatabaseSpec databaseSpec) {
        return dataApiManager.getDatabaseO(databaseSpec)
                .getOrElse(() -> MasterSlaveContextHolder
                        .withPolicy(MasterSlavePolicy.RW_M, () -> createDatabase(databaseSpec))
                );
    }

    private Database createDatabase(UserDatabaseSpec databaseSpec) {
        try {
            return dataApiManager.createDatabase(databaseSpec);
        } catch (DatabaseExistsException e) {
            // someone has created in async
            return dataApiManager.getDatabase(databaseSpec);
        }
    }

    protected Database applyDeltaWithRetries(DataApiUserId uid, DatabaseRef dbRef, Delta delta) {
        RuntimeException lastException = null;
        for (int delayMillis: RETRY_DELAYS) {
            try {
                return applyDelta(getDatabase(uid, dbRef), delta);
            } catch (OutdatedChangeException e) {
                lastException = e;
                logger.warn("Failed to execute: {}", e);
            }
            ThreadUtils.sleep(delayMillis);
        }
        throw lastException;
    }

    private Database applyDelta(Database database, Delta delta) {
        if (delta.changes.isEmpty()) {
            logger.debug("Delta is empty, don't save anything");
            return database;
        }
        return dataApiManager.applyDelta(database, RevisionCheckMode.PER_RECORD, delta);
    }
}
