package ru.yandex.chemodan.app.dataapi.apps.profile.address;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.Entity;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileRecordSupport;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileUtils;
import ru.yandex.commune.a3.action.result.pojo.ActionResultPojo;
import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.commune.protobuf5.annotation.ProtoMessage;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.geo.Coordinates;

/**
 * @author tolmalev
 */
@BenderBindAllFields
@ActionResultPojo
@ProtoMessage
public class Address extends ProfileRecordSupport implements Entity {

    public static final String TITLE = "title";

    public static final String LATITUDE = "latitude";
    public static final String LONGITUDE = "longitude";

    public static final String ADDRESS_LINE = "address_line";
    public static final String ADDRESS_LINE_SHORT = "address_line_short";

    public static final String CREATED = "created";
    public static final String MODIFIED = "modified";
    public static final String LAST_USED = "last_used";

    public static final String TAGS = "tags";
    public static final String MINED_ATTRIBUTES = "mined_attributes";

    public static final String ENTRANCE_NUMBER = "entrance_number";
    public static final String CUSTOM_METADATA = "custom_metadata";

    private static final String DEFAULT_TITLE = "no title";


    @BenderPart(name = TITLE, strictName = true)
    @ProtoField(n = 1)
    private String title = DEFAULT_TITLE;

    @ProtoField(n = 2)
    private double latitude;
    @ProtoField(n = 3)
    private double longitude;

    @BenderPart(name = "address_id", strictName = true)
    @ProtoField(n = 4)
    // optional only for parsing incoming requests
    private Option<String> addressId;

    @BenderPart(name = ADDRESS_LINE, strictName = true)
    @ProtoField(n = 5)
    private Option<String> addressLine = Option.empty();
    @BenderPart(name = ADDRESS_LINE_SHORT, strictName = true)
    @ProtoField(n = 6)
    private Option<String> addressLineShort = Option.empty();

    @BenderPart(name = CREATED, strictName = true)
    @ProtoField(n = 7)
    private Option<Instant> created = Option.empty();
    @BenderPart(name = MODIFIED, strictName = true)
    @ProtoField(n = 8)
    private Option<Instant> modified = Option.empty();
    @BenderPart(name = LAST_USED, strictName = true)
    @ProtoField(n = 9)
    private Option<Instant> lastUsed = Option.empty();

    @BenderPart(name = TAGS, strictName = true)
    @ProtoField(n = 10)
    private ListF<String> tags = Cf.arrayList();
    @BenderPart(name = MINED_ATTRIBUTES, strictName = true)
    @ProtoField(n = 11)
    private ListF<String> minedAttributes = Cf.arrayList();

    @BenderPart(name = ENTRANCE_NUMBER, strictName = true)
    @ProtoField(n = 12)
    private Option<String> entranceNumber = Option.empty();
    @BenderPart(name = CUSTOM_METADATA, strictName = true)
    @ProtoField(n = 13)
    private Option<String> customMetadata = Option.empty();

    public Address(Option<String> addressId, String title, Coordinates coordinates,
            Option<String> addressLine, Option<String> addressLineShort, Option<Instant> created,
            Option<Instant> modified,
            Option<Instant> lastUsed, ListF<String> tags, ListF<String> minedAttributes,
            Option<String> entranceNumber, Option<String> customMetadata)
    {
        this.title = title;
        this.latitude = coordinates.getLatitude();
        this.longitude = coordinates.getLongitude();
        this.addressId = addressId;
        this.addressLine = addressLine;
        this.addressLineShort = addressLineShort;
        this.created = created;
        this.modified = modified;
        this.lastUsed = lastUsed;
        this.tags = tags;
        this.minedAttributes = minedAttributes;
        this.entranceNumber = entranceNumber;
        this.customMetadata = customMetadata;
    }

    public Address(String addressId) {
        this.addressId = Option.of(addressId);
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public double getLatitude() {
        return latitude;
    }

    public void setLatitude(double latitude) {
        this.latitude = latitude;
    }

    public double getLongitude() {
        return longitude;
    }

    public void setLongitude(double longitude) {
        this.longitude = longitude;
    }

    public Option<String> getAddressId() {
        return addressId;
    }

    public void setAddressId(Option<String> addressId) {
        this.addressId = addressId;
    }

    public Option<String> getAddressLine() {
        return addressLine;
    }

    public void setAddressLine(Option<String> addressLine) {
        this.addressLine = addressLine;
    }

    public Option<String> getAddressLineShort() {
        return addressLineShort;
    }

    public void setAddressLineShort(Option<String> addressLineShort) {
        this.addressLineShort = addressLineShort;
    }

    public Option<Instant> getCreated() {
        return created;
    }

    public void setCreated(Option<Instant> created) {
        this.created = created;
    }

    public Option<Instant> getModified() {
        return modified;
    }

    public void setModified(Option<Instant> modified) {
        this.modified = modified;
    }

    public Option<Instant> getLastUsed() {
        return lastUsed;
    }

    public void setLastUsed(Option<Instant> lastUsed) {
        this.lastUsed = lastUsed;
    }

    public ListF<String> getTags() {
        return tags;
    }

    public void setTags(ListF<String> tags) {
        this.tags = tags;
    }

    public ListF<String> getMinedAttributes() {
        return minedAttributes;
    }


    public static Address fromDataRecord(DataRecord record) {
        return fromData(record.getRecordId(), record.getData());
    }

    public static Address fromData(String recordId, MapF<String, DataField> data) {
        return new Address(
                Option.of(recordId),
                data.getOrElse(TITLE, DataField.string(DEFAULT_TITLE)).stringValue(),
                new Coordinates(
                    data.getOrElse(LATITUDE, DataField.decimal(0)).decimalValue(),
                    data.getOrElse(LONGITUDE, DataField.decimal(0)).decimalValue()
                ),
                data.getO(ADDRESS_LINE).map(DataField.stringValueF()),
                data.getO(ADDRESS_LINE_SHORT).map(DataField.stringValueF()),

                data.getO(CREATED).map(DataField.timestampValueF()),
                data.getO(MODIFIED).map(DataField.timestampValueF()),
                data.getO(LAST_USED).map(DataField.timestampValueF()),

                Cf.arrayList(extractStringList(data.getO(TAGS))),
                Cf.arrayList(extractStringList(data.getO(MINED_ATTRIBUTES))),

                data.getO(ENTRANCE_NUMBER).map(DataField.stringValueF()),
                data.getO(CUSTOM_METADATA).map(DataField.stringValueF())
        );
    }

    @Override
    public MapF<String, DataField> toDataMap() {
        MapF<String, DataField> result = Cf.map();

        result = result.plus1(TITLE, DataField.string(title));
        result = result.plus1(LATITUDE, DataField.decimal(latitude));
        result = result.plus1(LONGITUDE, DataField.decimal(longitude));

        result = plusNonEmptyStringO(result, ADDRESS_LINE, addressLine);
        result = plusNonEmptyStringO(result, ADDRESS_LINE_SHORT, addressLineShort);

        result = plusTimestampO(result, CREATED, created);
        result = plusTimestampO(result, MODIFIED, modified);
        result = plusTimestampO(result, LAST_USED, lastUsed);

        result = plusList(result, TAGS, tags);
        result = plusList(result, MINED_ATTRIBUTES, minedAttributes);

        result = plusNonEmptyStringO(result, ENTRANCE_NUMBER, entranceNumber);
        result = plusNonEmptyStringO(result, CUSTOM_METADATA, customMetadata);

        return result;
    }

    public Address withId(String id) {
        return new Address(
                Option.of(id), title, new Coordinates(latitude, longitude), addressLine, addressLineShort, created,
                modified, lastUsed, tags, minedAttributes, entranceNumber, customMetadata
        );
    }

    public Address withTags(ListF<String> tags) {
        return new Address(
                Option.of(recordId()), title, new Coordinates(latitude, longitude), addressLine, addressLineShort,
                created, modified, lastUsed, tags, minedAttributes, entranceNumber, customMetadata
        );
    }

    @Override
    public CollectionRef collectionRef() {
        return ProfileUtils.ADDRESSES_COL_REF;
    }

    @Override
    public String recordId() {
        return getAddressId().get();
    }
}
