package ru.yandex.chemodan.app.dataapi.apps.profile.address;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.misc.test.Assert;

/**
 * @author Denis Bakharev
 */
@ActivateDataApiEmbeddedPg
public class AddressDataApiDaoIntegrationTest extends DataApiTestSupport {
    private DataApiUserId uid;
    private final Address address;

    public AddressDataApiDaoIntegrationTest() {
        address = new Address("id");
        address.setCreated(Option.of(Instant.now()));
        address.setLastUsed(Option.of(Instant.now()));
        address.setModified(Option.of(Instant.now()));
    }

    @Before
    public void before() {
        uid = createRandomCleanUser();
    }

    @Autowired
    public AddressDataApiDao addressDao;

    @Test
    public void updateAddress_WithDecreasedTagsCount_MustUpdateAddressInStorage() {
        address.getTags().add("first");
        address.getTags().add("delete me");

        addressDao.insertAddress(uid, address);

        address.getTags().remove(1);

        addressDao.updateAddress(uid, address);

        assertAddressInDatabaseEquals(address);
    }

    @Test
    public void updateAddress_WithIncreasedTagsCount_MustUpdateAddressInStorage() {
        address.getTags().add("update me");

        addressDao.insertAddress(uid, address);

        address.getTags().clear();
        address.getTags().add("new 1");
        address.getTags().add("new 2");

        addressDao.updateAddress(uid, address);

        assertAddressInDatabaseEquals(address);
    }

    @Test
    public void updateAddress_WithChangedFields_MustUpdateAddressInStorage() {
        address.setTitle("title1");
        address.setAddressLine(Option.of("addressLine"));

        addressDao.insertAddress(uid, address);

        address.setTitle("title2");
        address.setAddressLine(Option.empty());
        address.setAddressLineShort(Option.of("addressLineShort"));

        addressDao.updateAddress(uid, address);

        assertAddressInDatabaseEquals(address);
    }

    @Test
    public void setAddress_withEmptyShortAddressLine() {
        address.setAddressLineShort(Option.of(""));
        address.setAddressLine(Option.of(""));

        addressDao.setAddress(uid, address);

        Address savedAddress = addressDao.getEntityByUidAndId(uid, address.getAddressId().get());
        Assert.none(savedAddress.getAddressLineShort());
        Assert.none(savedAddress.getAddressLine());
    }

    private void assertAddressInDatabaseEquals(Address address) {
        Address updatedAddress = addressDao.getEntityByUidAndId(uid, address.getAddressId().get());

        Assert.equals(address, updatedAddress);
    }
}
