package ru.yandex.chemodan.app.dataapi.apps.profile.address;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordIdGenerator;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.misc.geo.Coordinates;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class AddressManager {
    private static final Logger logger = LoggerFactory.getLogger(AddressManager.class);
    private final AddressDataApiDao addressDao;
    private final GeocoderHelper geocoderHelper;

    public AddressManager(AddressDataApiDao addressDao, GeocoderHelper geocoderHelper) {
        this.addressDao = addressDao;
        this.geocoderHelper = geocoderHelper;
    }

    public void patchAddress(DataApiUserId user, String addressId, AddressPatch patch) {
        addressDao.patchAddress(user, addressId, patch);
    }

    public Address findUserAddress(DataApiUserId user, String addressId) {
        return addressDao.getEntityByUidAndId(user, addressId);
    }

    public Address createAddress(DataApiUserId uid, Address address) {
        if (!address.getAddressId().isPresent()) {
            address.setAddressId(Option.of(RecordIdGenerator.generateId()));
        }

        Option<Instant> nowO = Option.of(Instant.now());
        address.setCreated(nowO);
        address.setModified(nowO);
        address.setLastUsed(nowO);

        address.getMinedAttributes().clear();

        addressDao.insertAddress(uid, address);

        return address;
    }

    public Address putAddressIgnoringMinedAttributes(DataApiUserId uid, Address newAddress) {
        String addressId = newAddress.getAddressId().get();
        Option<Address> existedAddressO = addressDao.getEntityByUidAndIdO(uid, addressId);

        Option<Instant> nowO = Option.of(Instant.now());
        if (!existedAddressO.isPresent()) {
            if (!newAddress.getAddressId().isPresent()) {
                newAddress.setAddressId(Option.of(RecordIdGenerator.generateId()));
            }

            newAddress.setCreated(nowO);
            newAddress.setLastUsed(nowO);
        } else {
            Address existedAddress = existedAddressO.get();
            newAddress.setCreated(existedAddress.getCreated());
            newAddress.setLastUsed(existedAddress.getLastUsed());
        }

        newAddress.setModified(nowO);
        newAddress.getMinedAttributes().clear();

        addressDao.setAddress(uid, newAddress);

        return newAddress;
    }

    public void deleteAddress(DataApiUserId uid, String addressId) {
        addressDao.deleteEntity(uid, addressId);
    }

    public void addTags(DataApiUserId uid, String addressId, ListF<String> tags) {
        Address address = addressDao.getEntityByUidAndId(uid, addressId);
        tags = tags.filter((tag) -> !address.getTags().containsTs(tag));

        address.getTags().addAll(tags);

        updateAddress(uid, address);
    }

    public void removeTags(DataApiUserId uid, String addressId, ListF<String> tags) {
        Address address = addressDao.getEntityByUidAndId(uid, addressId);
        address.getTags().removeAllTs(tags);

        updateAddress(uid, address);
    }

    public void touchAddress(DataApiUserId uid, String addressId) {
        Address address = addressDao.getEntityByUidAndId(uid, addressId);
        address.setLastUsed(Option.of(Instant.now()));

        updateAddress(uid, address);
    }

    public void updateAddress(DataApiUserId uid, Address address) {
        address.setModified(Option.of(Instant.now()));

        addressDao.updateAddress(uid, address);
    }

    public void setAddressLinesFromGeocoder(DataApiUserId uid, String addressRecordId, long addressRecordRevision) {
        Option<DataRecord> dataRecordO = addressDao.getRecord(uid, addressRecordId);
        if (!dataRecordO.isPresent()) {
            logger.info("DataRecord with recordId = {} does not exist", addressRecordId);
            return;
        }

        DataRecord dataRecord = dataRecordO.get();
        if (dataRecord.rev != addressRecordRevision) {
            logger.info("We expect DataRecord with rev = {}, but got DataRecord with rev = {}",
                        addressRecordRevision,
                        dataRecord.rev);
            return;
        }

        Address address = Address.fromDataRecord(dataRecord);
        Coordinates coordinates = new Coordinates(address.getLatitude(), address.getLongitude());

        Check.isTrue(!address.getAddressLine().isPresent() || !address.getAddressLineShort().isPresent(),
                "address.addressLine or address.addressLineShort must be empty");
        Option<AddressLines> addressLines = geocoderHelper.findAddressLines(coordinates);

        if (!addressLines.isPresent()) {
            logger.info("Could not get any AddressLines from coordinates = {}", coordinates);
            return;
        }

        if (!address.getAddressLine().isPresent()) {
            // we should set both addressLine and addressLineShort
            setAddressLine(address, addressLines);
            setAddressLineShort(address, addressLines);
        }

        if (!address.getAddressLineShort().isPresent()) {
            // we should set only addressLineShort
            setAddressLineShort(address, addressLines);
        }

        updateAddress(uid, address);
    }

    private void setAddressLine(Address address, Option<AddressLines> addressLines) {
        address.setAddressLine(Option.of(addressLines.get().getAddressLine()));
        addToMinedAttributesIfNotContains(address, Address.ADDRESS_LINE);
    }

    private void setAddressLineShort(Address address, Option<AddressLines> addressLines) {
        address.setAddressLineShort(Option.of(addressLines.get().getAddressLineShort()));
        addToMinedAttributesIfNotContains(address, Address.ADDRESS_LINE_SHORT);
    }

    private void addToMinedAttributesIfNotContains(Address address, String fieldName) {
        if (!address.getMinedAttributes().containsTs(fieldName)) {
            address.getMinedAttributes().add(fieldName);
        }
    }
}
