package ru.yandex.chemodan.app.dataapi.apps.profile.address;

import org.joda.time.DateTimeUtils;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaValidationException;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.misc.test.Assert;

/**
 * @author Denis Bakharev
 */
@ActivateDataApiEmbeddedPg
public class AddressManagerIntegrationTest extends DataApiTestSupport {

    private final Instant now = Instant.now();
    private DataApiUserId uid;

    @Autowired
    public AddressManager addressManager;

    private final Address address;
    private String addressId;

    public AddressManagerIntegrationTest() {
        address = new Address("id");
        addressId = address.getAddressId().get();
    }

    @Before
    public void before() {
        uid = createRandomCleanUser();

        DateTimeUtils.setCurrentMillisProvider(now::getMillis);
    }

    @After
    public void after() {
        DateTimeUtils.setCurrentMillisSystem();
    }

    @Test
    public void createAddress_AddressWithoutDateTimes_SetAllDateTimesToNow() {
        Address address = new Address("id");
        Address returnedAddress = addressManager.createAddress(uid, address);

        Assert.equals(now, returnedAddress.getCreated().get());
        Assert.equals(now, returnedAddress.getModified().get());
        Assert.equals(now, returnedAddress.getLastUsed().get());
    }

    @Test
    public void createAddress_BothAddressLinesNotSet_AddressLinesUpdatedFromGeocoder() {
        address.setLatitude(55.75438);
        address.setLongitude(37.621462);
        addressManager.createAddress(uid, address);

        Assert.sizeIs(1, bazingaStub.tasksWithParams);
        bazingaStub.executeTasks(applicationContext);

        Address updatedAddress = addressManager.findUserAddress(uid, addressId);
        Assert.isTrue(updatedAddress.getAddressLine().get().startsWith("Россия, Москва, "));
        Assert.isTrue(updatedAddress.getAddressLine().get().endsWith(", Красная площадь, 3"));
        Assert.some("Красная площадь, 3", updatedAddress.getAddressLineShort());
        Assert.in(Address.ADDRESS_LINE, updatedAddress.getMinedAttributes());
        Assert.in(Address.ADDRESS_LINE_SHORT, updatedAddress.getMinedAttributes());
    }

    @Test
    public void deleteAddress_ExistedAddress_DeletesAddressFromDb() {
        addressManager.createAddress(uid, address);

        addressManager.deleteAddress(uid, addressId);
    }

    @Test
    public void deleteAddress_NotExistedAddress_ThrowsDeltaValidationException() {
        Assert.assertThrows(() -> addressManager.deleteAddress(uid, addressId),
                            DeltaValidationException.class);
    }

    @Test
    public void touchAddress_ExistedAddress_UpdatesLastUsed() {
        addressManager.createAddress(uid, address);

        Instant newDate = Instant.now().plus(Duration.standardDays(1));
        DateTimeUtils.setCurrentMillisProvider(newDate::getMillis);

        addressManager.touchAddress(uid, addressId);

        Address touchedAddress = addressManager.findUserAddress(uid, addressId);
        Assert.equals(newDate, touchedAddress.getLastUsed().get());
    }

    @Test
    public void putAddressIgnoringMinedAttributes_ExistedAddress_AddressWithoutMinedAttributes() {
        addressManager.createAddress(uid, address);

        putAddressAndAssertEmptyMinedAttributes(address);
    }

    private void putAddressAndAssertEmptyMinedAttributes(Address address) {
        address.getMinedAttributes().add("MinedAttribute");
        addressManager.putAddressIgnoringMinedAttributes(uid, address);

        Address newAddress = addressManager.findUserAddress(uid, address.getAddressId().get());
        Assert.isEmpty(newAddress.getMinedAttributes());
    }

    @Test
    public void removeTags_RemoveOnlyLastTags_RemovesTagsFromAddress() {
        removeTags(Cf.list("tag1", "tag2", "tag3"), Cf.list("tag2", "tag3"));
    }

    @Test
    public void removeTags_RemoveOnlyFirstTags_RemovesTagsFromAddress() {
        removeTags(Cf.list("tag1", "tag2", "tag3"), Cf.list("tag1", "tag2"));
    }

    @Test
    public void removeTags_RemoveTags_RemovesTagsFromAddress() {
        removeTags(Cf.list("tag1", "tag2", "tag3", "tag4", "tag5"), Cf.list("tag1", "tag3", "tag5"));
    }

    private void removeTags(ListF<String> initialTags, ListF<String> tagsForRemove) {
        for (String tag : initialTags) {
            address.getTags().add(tag);
        }

        addressManager.createAddress(uid, address);
        addressManager.removeTags(uid, addressId, tagsForRemove);
        Address addressWithoutTag = addressManager.findUserAddress(uid, addressId);

        for (String removedTag : tagsForRemove) {
            Assert.notIn(removedTag, addressWithoutTag.getTags());
        }

        for (String notRemovedTag : initialTags.filterNot(tagsForRemove::containsTs)) {
            Assert.in(notRemovedTag, addressWithoutTag.getTags());
        }
    }

    @Test
    public void addTags_AddressWithoutTags_AddsTagToAddress() {
        addressManager.createAddress(uid, address);

        addressManager.addTags(uid, addressId, Cf.list("tag1"));

        Address addressWithTag = addressManager.findUserAddress(uid, addressId);
        Assert.in("tag1", addressWithTag.getTags());
    }

    @Test
    public void patchAddress_ExistedAddress_UpdatesFieldsInAddress() {
        addressManager.createAddress(uid, address);

        FieldChange fieldChange = FieldChange.put(Address.TITLE, DataField.string("new title"));
        AddressPatch patch = new AddressPatch(Cf.list(fieldChange));

        addressManager.patchAddress(uid, addressId, patch);

        Address patchedAddress = addressManager.findUserAddress(uid, addressId);
        Assert.equals("new title", patchedAddress.getTitle());
    }
}
