package ru.yandex.chemodan.app.dataapi.apps.profile.address;

import org.junit.Test;
import org.mockito.ArgumentCaptor;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileUtils;
import ru.yandex.chemodan.app.dataapi.apps.profile.test.ProfileUnitTestBase;
import ru.yandex.misc.test.Assert;

import static org.mockito.Mockito.any;
import static org.mockito.Mockito.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

/**
 * @author Denis Bakharev
 */
public class AddressManagerUnitTest extends ProfileUnitTestBase {

    private final DataRecordId rid;
    private final DataApiUserId uid;
    private final AddressDataApiDao storageMock;
    private final GeocoderHelper geocoderHelperMock;
    private final AddressLines addressLines;

    public AddressManagerUnitTest() {
        rid = ProfileUtils.ADDRESSES_COL_REF
                .consRecordRef("recId")
                .toRecordId(ProfileUtils.ADDRESSES_COL_REF.dbRef().consHandle("handle"));
        uid = getUserId();
        storageMock = mock(AddressDataApiDao.class);
        geocoderHelperMock = mock(GeocoderHelper.class);
        addressLines = new AddressLines("geocoderAddressLine", "geocoderAddressLineShort");
        when(geocoderHelperMock.findAddressLines(any())).thenReturn(Option.of(addressLines));
    }

    @Test
    public void execute_DataRecordRevChanged_DoNothing() throws Exception {
        setupDataRecordReturn(Option.of(new DataRecord(uid, rid, 101, Cf.map())));

        changeAddressWithRevision(100);

        assertDoNothing();
    }

    @Test
    public void execute_DataRecordNotFound_DoNothing() throws Exception {
        setupDataRecordReturn(Option.empty());

        changeAddressWithRevision(100);

        assertDoNothing();
    }

    @Test
    public void execute_BothAddressLinesNotSet_SetBothAddressLinesFromGeocoderAndAddToMinedAttributes() throws
            Exception
    {
        Address sourceAddress = new Address(rid.recordId());
        setupDataRecordReturn(Option.of(new DataRecord(uid, rid, 100, sourceAddress.toDataMap())));

        changeAddressWithRevision(100);

        Address capturedAddress = captureUpdateAddressInvocation();
        Assert.equals(addressLines.getAddressLine(), capturedAddress.getAddressLine().get());
        Assert.equals(addressLines.getAddressLineShort(), capturedAddress.getAddressLineShort().get());
        Assert.assertContains(capturedAddress.getMinedAttributes(), Address.ADDRESS_LINE);
        Assert.assertContains(capturedAddress.getMinedAttributes(), Address.ADDRESS_LINE_SHORT);
    }

    @Test
    public void execute_AddressLineShortNotSet_SetAddressLineShortFromGeocoderAndAddToMinedAttributes() throws
            Exception
    {
        Address address = new Address(rid.recordId());
        address.setAddressLine(Option.of("addressLineTest"));

        setupDataRecordReturn(Option.of(new DataRecord(uid, rid, 100, address.toDataMap())));

        changeAddressWithRevision(100);

        Address capturedAddress = captureUpdateAddressInvocation();
        Assert.equals(capturedAddress.getAddressLine().get(), "addressLineTest");
        Assert.equals(capturedAddress.getAddressLineShort().get(), addressLines.getAddressLineShort());
        Assert.in(Address.ADDRESS_LINE_SHORT, capturedAddress.getMinedAttributes());
        Assert.notIn(Address.ADDRESS_LINE, capturedAddress.getMinedAttributes());
    }

    @Test
    public void execute_AddressLineNotSet_SetBothAddressLinesFromGeocoderAndAddToMinedAttributes() throws
            Exception
    {
        Address address = new Address(rid.recordId());
        address.setAddressLineShort(Option.of("addressLineShortTest"));

        setupDataRecordReturn(Option.of(new DataRecord(uid, rid, 100, address.toDataMap())));

        changeAddressWithRevision(100);

        Address capturedAddress = captureUpdateAddressInvocation();
        Assert.equals(addressLines.getAddressLine(), capturedAddress.getAddressLine().get());
        Assert.equals(addressLines.getAddressLineShort(), capturedAddress.getAddressLineShort().get());
        Assert.in(Address.ADDRESS_LINE_SHORT, capturedAddress.getMinedAttributes());
        Assert.in(Address.ADDRESS_LINE, capturedAddress.getMinedAttributes());

    }

    private void setupDataRecordReturn(Option<DataRecord> dataRecord) {
        when(storageMock.getRecord(uid, rid.recordId()))
                .thenReturn(dataRecord);
    }

    private void changeAddressWithRevision(long dataRecordRev) {
        AddressManager addressesManager = new AddressManager(storageMock, geocoderHelperMock);
        addressesManager.setAddressLinesFromGeocoder(uid, rid.recordId(), dataRecordRev);
    }

    private void assertDoNothing() {
        verify(storageMock, never()).updateAddress(eq(uid), any());
    }

    private Address captureUpdateAddressInvocation() {
        ArgumentCaptor<Address> addressCaptor = ArgumentCaptor.forClass(Address.class);
        verify(storageMock).updateAddress(eq(uid), addressCaptor.capture());

        return addressCaptor.getValue();
    }
}
