package ru.yandex.chemodan.app.dataapi.apps.profile.cache;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import net.spy.memcached.CachedData;
import net.spy.memcached.MemcachedClient;
import net.spy.memcached.transcoders.Transcoder;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.profile.full.FullUserProfile;
import ru.yandex.chemodan.app.dataapi.utils.memcached.MemcachedClients;
import ru.yandex.commune.protobuf5.Protobuf5Serializer;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class MemcachedFullProfileCache implements FullProfileCache {
    private static final Logger logger = LoggerFactory.getLogger(MemcachedFullProfileCache.class);

    private final ExecutorService executor = Executors.newFixedThreadPool(100);

    private final Transcoder<FullUserProfile> transcoder = new FullProfileTranscoder();

    private final MemcachedClients memcachedClients;

    private final int expireSeconds;
    private final int getTimeoutMillis;

    public MemcachedFullProfileCache(MemcachedClients memcachedClients, int expireSeconds, int getTimeoutMillis) {
        this.expireSeconds = expireSeconds;
        this.getTimeoutMillis = getTimeoutMillis;
        this.memcachedClients = memcachedClients;
    }

    @Override
    public Option<FullUserProfile> getFromCache(DataApiUserId uid) {
        if (!mainMemcachedClient().isPresent()) {
            return Option.empty();
        }
        try {
            Option<FullUserProfile> result = Option.ofNullable(mainMemcachedClient().get()
                    .asyncGet(getKey(uid), transcoder)
                    .get(getTimeoutMillis, TimeUnit.MILLISECONDS));
            logger.debug("Search in cache for {}: {}", uid, result.isPresent() ? "found" : "not_found");
            return result;
        } catch (Exception e) {
            logger.warn("Can't get from cache: {}", e);
            return Option.empty();
        }
    }

    @Override
    public void putInCacheAsync(DataApiUserId uid, FullUserProfile profile) {
        executor.submit(() -> {
            if (!mainMemcachedClient().isPresent()) {
                return;
            }
            logger.debug("Put in cache for {}", uid);
            try {
                mainMemcachedClient().get().set(getKey(uid), expireSeconds, profile, transcoder);
            } catch (Exception e) {
                logger.error("Can't put in cache: {}", e);
            }
        });
    }

    private Option<MemcachedClient> mainMemcachedClient() {
        return memcachedClients.mainMemcachedClient;
    }

    @Override
    public void invalidateForUserAsync(DataApiUserId uid) {
        logger.debug("Invalidating cache for {}", uid);
        for (MemcachedClient client : memcachedClients.allClients) {
            try {
                client.delete(getKey(uid));
            } catch (Exception e) {
                logger.error("Failed to remove from cache: {}", e);
            }
        }
    }

    public String getKey(DataApiUserId uid) {
        return uid.toString();
    }

    private static final class FullProfileTranscoder implements Transcoder<FullUserProfile> {

        Protobuf5Serializer<FullUserProfile> S = Protobuf5Serializer.cons(FullUserProfile.class);

        @Override
        public boolean asyncDecode(CachedData d) {
            return false;
        }

        @Override
        public CachedData encode(FullUserProfile o) {
            return new CachedData(0, S.serialize(o), CachedData.MAX_SIZE);
        }

        @Override
        public FullUserProfile decode(CachedData d) {
            return S.deserialize(d.getData());
        }

        @Override
        public int getMaxSize() {
            return CachedData.MAX_SIZE;
        }
    }
}
