package ru.yandex.chemodan.app.dataapi.apps.profile.events;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileManagerBase;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileUtils;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.RoundRobinCounter;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;
import ru.yandex.misc.time.clock.Clock;

/**
 * @author tolmalev
 */
public class ActualEventsManager extends ProfileManagerBase implements MonicaContainer {
    private static final Logger logger = LoggerFactory.getLogger(ActualEventsManager.class);

    private final AsyncEventRemover asyncEventRemover;

    @GroupByDefault
    @MonicaMetric(description = "Not found events, marked for removing in 5 minutes")
    private RoundRobinCounter notFoundEvents = new RoundRobinCounter(Duration.standardMinutes(5));
    private Clock clock = Clock.DEFAULT;

    public ActualEventsManager(DataApiManager dataApiManager, DiskDataSource diskDataSource,
            AsyncEventRemover asyncEventRemover)
    {
        super(dataApiManager, diskDataSource);
        this.asyncEventRemover = asyncEventRemover;
    }

    public void removeOutdatedEvent(DataApiUserId uid, EventType eventType, String recordId) {
        Option<Event> eventO = getEventO(uid, eventType, recordId);
        if (!eventO.isPresent()) {
            logger.warn("Event not found");
            notFoundEvents.inc();
            return;
        }

        Event event = eventO.get();
        if (isOutdated(event)) {
            logger.info("Event is outdated, removing at {}, now: {}. event: {}", event.getEvictionTime(),
                    clock.now(), event);
            removeEvent(uid, eventType, recordId);
        } else {
            logger.warn("Event mustn't be removed now: {}", event);
            asyncEventRemover.scheduleEventRemoving(uid, recordId, event);
        }
    }

    public void removeEvent(DataApiUserId uid, EventType eventType, String recordId) {
        logger.info("Removing event. uid {}, type: {}, id: {}", uid, eventType, recordId);
        Delta delta = new Delta(RecordChange.delete(eventType.colRef, recordId));
        applyDeltaWithRetries(uid, ProfileUtils.EVENTS_DB_REF, delta);
    }

    public boolean isOutdated(Event event) {
        return clock.now().isAfter(event.getEvictionTime());
    }

    public Option<Event> getEventO(DataApiUserId uid, EventType eventType, String recordId) {
        return getRecord(uid, eventType.colRef.consRecordRef(recordId))
                .map(eventType::parse);
    }

    public void setClockForTests(Clock clock) {
        this.clock = clock;
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "actual-events-manager",
                new MetricName("dataapi", "profile", "actual-events"),
                "Actual events (flights, hotels, etc)"
        );
    }
}
