package ru.yandex.chemodan.app.dataapi.apps.profile.events;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseMeta;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileUtils;
import ru.yandex.chemodan.app.dataapi.apps.profile.events.flights.Flight;
import ru.yandex.chemodan.app.dataapi.apps.profile.events.flights.ProfileFlightsManagerTest;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.clock.FixedClock;

import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode.PER_RECORD;

/**
 * @author tolmalev
 */
public class ActualEventsManagerTest {

    private DataApiManager dataApiManager = Mockito.mock(DataApiManager.class);

    private DiskDataSource diskDataSource = Mockito.mock(DiskDataSource.class);

    private BazingaTaskManager bazingaTaskManager = Mockito.mock(BazingaTaskManager.class);

    private ActualEventsManager eventsManager = new ActualEventsManager(dataApiManager, diskDataSource,
            new AsyncEventRemover(bazingaTaskManager));

    private FixedClock clock = new FixedClock(new Instant(0));

    private static DataApiPassportUserId sampleUid = new DataApiPassportUserId(100L);
    private static String sampleRecordId = "test-flight-1";
    private static Flight sampleFlight = ProfileFlightsManagerTest.sampleFlight();

    public ActualEventsManagerTest() {
        Mockito
                .when(dataApiManager.getDatabaseO(new UserDatabaseSpec(sampleUid, ProfileUtils.EVENTS_DB_REF))
                ).thenReturn(Option.of(new Database(sampleUid,
                    ProfileUtils.EVENTS_DB_REF.consHandle("handle"), 0, mock(DatabaseMeta.class))));

        Mockito
                .when(dataApiManager.getRecord(Mockito.<DataApiUserId>any(), any()))
                .thenReturn(Option.empty());

        DatabaseHandle handle = ProfileUtils.FLIGHTS_COL_REF.dbRef().consHandle("handle");
        DataRecordId recordId = ProfileUtils.FLIGHTS_COL_REF.consRecordRef(sampleRecordId)
                .toRecordId(handle);
        Mockito
                .when(dataApiManager.getRecord(sampleUid, ProfileUtils.FLIGHTS_COL_REF.consRecordRef(sampleRecordId))
                ).thenReturn(Option.of(new DataRecord(
                sampleUid, recordId, 0, sampleFlight.toDataMap()
        )));

        eventsManager.setClockForTests(clock);
    }

    @Test
    public void outdated() {
        clock.increment(Duration.millis(1000));
        Assert.isTrue(eventsManager.isOutdated(new StubEvent(new Instant(999))));
        Assert.isFalse(eventsManager.isOutdated(new StubEvent(new Instant(1000))));
        Assert.isFalse(eventsManager.isOutdated(new StubEvent(new Instant(1001))));
    }

    @Test
    public void findEvent() {
        Assert.some(
                sampleFlight.withId(sampleRecordId),
                eventsManager.getEventO(sampleUid, EventType.FLIGHT, sampleRecordId));
    }

    @Test
    public void removeEvent() {
        eventsManager.removeEvent(sampleUid, EventType.FLIGHT, sampleRecordId);

        verify(dataApiManager, times(1)).applyDelta(any(), eq(PER_RECORD), any());
    }

    @Test
    public void removeOutdatedEvent() {
        clock.increment(Duration.millis(sampleFlight.getEvictionTime().getMillis() + 1000));
        eventsManager.removeOutdatedEvent(sampleUid, EventType.FLIGHT, sampleRecordId);

        verify(dataApiManager, times(1)).applyDelta(any(), eq(PER_RECORD), any());
    }

    @Test
    public void notRemoveNotOutdatedEvent() {
        clock.increment(Duration.millis(sampleFlight.getEvictionTime().getMillis() - 1000));
        eventsManager.removeOutdatedEvent(sampleUid, EventType.FLIGHT, sampleRecordId);

        verify(dataApiManager, times(0)).applyDelta(any(), eq(PER_RECORD), any());
        verify(bazingaTaskManager, times(1)).schedule(any(), any());
    }

    @Test
    public void removeNotExistingEvent() {
        eventsManager.removeOutdatedEvent(sampleUid, EventType.FLIGHT, "bad record id");

        verify(dataApiManager, times(0)).applyDelta(any(), eq(PER_RECORD), any());
        verify(bazingaTaskManager, times(0)).schedule(any(), any());
    }

    private static final class StubEvent implements Event {

        private final Instant evictionTime;

        StubEvent(Instant evictionTime) {
            this.evictionTime = evictionTime;
        }

        @Override
        public Instant getStartTime() {
            return evictionTime;
        }

        @Override
        public Option<Instant> getEndTime() {
            return Option.empty();
        }

        @Override
        public Instant getEvictionTime() {
            return evictionTime;
        }

        @Override
        public EventType getEventType() {
            return null;
        }
    }
}
