package ru.yandex.chemodan.app.dataapi.apps.profile.events.flights;

import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.Entity;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileRecordSupport;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileUtils;
import ru.yandex.chemodan.app.dataapi.apps.profile.events.Event;
import ru.yandex.chemodan.app.dataapi.apps.profile.events.EventType;
import ru.yandex.commune.a3.action.result.pojo.ActionResultPojo;
import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.commune.protobuf5.annotation.ProtoMessage;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;

/**
 * @author tolmalev
 */
@ActionResultPojo
@BenderBindAllFields
@ProtoMessage
public class Flight extends ProfileRecordSupport implements Event, Entity {

    public static final String FLIGHT_NUMBER = "flight_number";
    public static final String CHECKIN_URL = "checkin_url";
    public static final String DATA_SOURCE = "data_source";

    public static final String AIRLINE_PREFIX = "airline_";
    public static final String DEPARTURE_PREFIX = "departure_";
    public static final String ARRIVAL_PREFIX = "arrival_";

    @BenderPart(name = "id", strictName = true)
    @ProtoField(n = 1)
    public final Option<String> uniqueFlightId;

    @ProtoField(n = 2)
    @BenderPart(name = FLIGHT_NUMBER, strictName = true)
    public final String flightNumber;

    @ProtoField(n = 3)
    @BenderPart(name = CHECKIN_URL, strictName = true)
    public final Option<String> checkinUrl;

    @ProtoField(n = 4)
    public final Option<Airline> airline;

    @ProtoField(n = 5)
    public final FlightPlace departure;
    @ProtoField(n = 6)
    public final Option<FlightPlace> arrival;

    @ProtoField(n = 7)
    @BenderPart(name = DATA_SOURCE, strictName = true)
    public final Option<String> dataSource;

    public Flight(Option<String> uniqueFlightId, String flightNumber, Option<String> checkinUrl,
            Option<Airline> airline, FlightPlace departure, Option<FlightPlace> arrival, Option<String> dataSource)
    {
        this.flightNumber = flightNumber;
        this.checkinUrl = checkinUrl;
        this.airline = airline;
        this.departure = departure;
        this.arrival = arrival;
        this.uniqueFlightId = uniqueFlightId;
        this.dataSource = dataSource;
    }

    public Flight(String flightNumber, Option<String> checkinUrl, Option<Airline> airline,
            FlightPlace departure, Option<FlightPlace> arrival, Option<String> dataSource)
    {
        this.flightNumber = flightNumber;
        this.checkinUrl = checkinUrl;
        this.airline = airline;
        this.departure = departure;
        this.arrival = arrival;
        this.dataSource = dataSource;
        this.uniqueFlightId = Option.empty();
    }

    public static Flight fromDataRecord(DataRecord record) {
        return fromData(record.getRecordId(), record.getData());
    }

    public static Flight fromData(String recordId, MapF<String, DataField> data) {
        return new Flight(
                Option.of(recordId),
                data.getTs(FLIGHT_NUMBER).stringValue(),
                data.getO(CHECKIN_URL).map(DataField::stringValue),
                Airline.parse(AIRLINE_PREFIX, data),
                FlightPlace.parse(DEPARTURE_PREFIX, data).get(),
                FlightPlace.parse(ARRIVAL_PREFIX, data),
                data.getO(DATA_SOURCE).map(DataField::stringValue));
    }

    @Override
    public MapF<String, DataField> toDataMap() {
        MapF<String, DataField> result = Cf
                .map(FLIGHT_NUMBER, DataField.string(flightNumber));

        result = plusStringO(result, CHECKIN_URL, checkinUrl);
        result = plusStringO(result, DATA_SOURCE, dataSource);

        result = addSubObject(result, AIRLINE_PREFIX, airline);
        result = addSubObject(result, DEPARTURE_PREFIX, departure);
        result = addSubObject(result, ARRIVAL_PREFIX, arrival);

        return result;
    }

    @Override
    public Instant getStartTime() {
        //Departure time is always defined
        return departure.time.get();
    }

    @Override
    public Instant getEvictionTime() {
        return getEndTime()
                .getOrElse(getStartTime())
                .plus(Duration.standardDays(1));
    }

    @Override
    public Option<Instant> getEndTime() {
        return arrival.flatMapO(place -> place.time);
    }

    @Override
    public EventType getEventType() {
        return EventType.FLIGHT;
    }

    public Flight withId(String id) {
        return new Flight(Option.of(id), flightNumber, checkinUrl, airline, departure, arrival, dataSource);
    }

    @Override
    public CollectionRef collectionRef() {
        return ProfileUtils.FLIGHTS_COL_REF;
    }

    @Override
    public String recordId() {
        return uniqueFlightId.get();
    }
}
