package ru.yandex.chemodan.app.dataapi.apps.profile.events.flights;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.apps.profile.ProfileRecordSupport;
import ru.yandex.commune.protobuf5.annotation.ProtoField;
import ru.yandex.commune.protobuf5.annotation.ProtoMessage;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;

/**
 * @author tolmalev
 */
@BenderBindAllFields
@ProtoMessage
public class FlightPlace extends ProfileRecordSupport {

    public static final String TIME = "time";
    public static final String TZ_OFFSET = "tzoffset";
    public static final String GEO_ID = "geo_id";
    public static final String CITY_NAME = "city_name";
    public static final String AIRPORT_PREFIX = "airport_";

    @ProtoField(n = 1)
    public final Option<Instant> time;
    @BenderPart(name = TZ_OFFSET, strictName = true)
    @ProtoField(n = 2)
    public final Option<Integer> tzOffset;
    @BenderPart(name = GEO_ID, strictName = true)
    @ProtoField(n = 3)
    public final Option<Integer> geoId;
    @ProtoField(n = 4)
    public final Option<Airport> airport;
    @ProtoField(n = 5)
    @BenderPart(name = CITY_NAME, strictName = true)
    public final Option<String> cityName;

    public FlightPlace(Option<Instant> time, Option<Integer> tzOffset, Option<Integer> geoId, Option<Airport> airport,
            Option<String> cityName)
    {
        this.time = time;
        this.tzOffset = tzOffset;
        this.geoId = geoId;
        this.airport = airport;
        // city_name is set only when geo isn't defined
        this.cityName = geoId.isPresent() ? Option.empty() : cityName;
    }

    public static Option<FlightPlace> parse(String prefix, MapF<String, DataField> data) {
        if (!data.containsKeyTs(prefix + TIME)) {
            return Option.empty();
        }
        return Option.of(parse(
                data.getO(prefix + TIME),
                data.getO(prefix + TZ_OFFSET),
                data.getO(prefix + GEO_ID),
                Airport.parse(prefix + AIRPORT_PREFIX, data),
                data.getO(prefix + CITY_NAME)
        ));
    }

    public static FlightPlace parse(Option<DataField> time, Option<DataField> timezone, Option<DataField> geoId,
            Option<Airport> airport, Option<DataField> cityName)
    {
        return new FlightPlace(
                time.map(DataField::timestampValue),
                timezone.map(DataField::integerValue).map(Long::intValue),
                geoId.map(DataField::integerValue).map(Long::intValue),
                airport,
                cityName.map(DataField::stringValue));
    }

    @Override
    public MapF<String, DataField> toDataMap() {
        MapF<String, DataField> result = Cf.map();

        result = plusTimestampO(result, TIME, time);
        result = plusIntegerO(result, TZ_OFFSET, tzOffset);
        result = plusIntegerO(result, GEO_ID, geoId);
        // city_name is set only when geo isn't defined
        if (!geoId.isPresent()) {
            result = plusStringO(result, CITY_NAME, cityName);
        }
        result = addSubObject(result, AIRPORT_PREFIX, airport);

        return result;
    }
}
